<?php
require_once __DIR__ . '/../../config/bot.php';

/**
 * کلاس اصلی بات تلگرام
 */
class TelegramBot {
    private $botToken;
    private $apiUrl;
    
    public function __construct() {
        $this->botToken = BotConfig::getBotToken();
        $this->apiUrl = "https://api.telegram.org/bot{$this->botToken}/";
    }
    
    /**
     * ارسال درخواست به API تلگرام
     */
    public function sendRequest($method, $data = []) {
        $url = $this->apiUrl . $method;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Content-Length: ' . strlen(json_encode($data))
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            error_log("Telegram API error: HTTP $httpCode - $response");
            return false;
        }
        
        $decoded = json_decode($response, true);
        
        if (!$decoded) {
            error_log("Failed to decode JSON response: $response");
            return false;
        }
        
        return $decoded;
    }
    
    /**
     * ارسال پیام متنی
     */
    public function sendMessage($chatId, $text, $replyMarkup = null, $replyKeyboard = null, $disableWebPagePreview = false, $parseMode = 'HTML') {
        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'disable_web_page_preview' => $disableWebPagePreview
        ];

        if (!empty($parseMode)) {
            $data['parse_mode'] = $parseMode;
        }
        
        // در تلگرام نمی‌توانیم همزمان inline keyboard و reply keyboard داشته باشیم
        // اگر reply keyboard داده شده و inline keyboard نیست، از reply keyboard استفاده می‌کنیم
        // در غیر این صورت از inline keyboard استفاده می‌کنیم
        if ($replyMarkup) {
            $data['reply_markup'] = $replyMarkup;
        } elseif ($replyKeyboard) {
            $data['reply_markup'] = $replyKeyboard;
        }
        
        return $this->sendRequest('sendMessage', $data);
    }
    
    /**
     * تنظیم Reply Keyboard برای یک کاربر (برای نمایش دکمه در پایین صفحه)
     * این کیبورد تا زمانی که حذف نشود، در پایین صفحه باقی می‌ماند
     */
    public function setReplyKeyboard($chatId, $replyKeyboard) {
        // برای تنظیم reply keyboard، باید یک پیام با reply keyboard ارسال کنیم
        // از متن خالی استفاده می‌کنیم تا پیام نامرئی باشد
        $data = [
            'chat_id' => $chatId,
            'text' => ' ',
            'reply_markup' => $replyKeyboard
        ];
        
        $result = $this->sendRequest('sendMessage', $data);
        
        // حذف پیام بعد از یک ثانیه (اختیاری - می‌توانید حذف کنید)
        if ($result && isset($result['result']['message_id'])) {
            // می‌توانیم پیام را بعداً حذف کنیم یا نگه داریم
            // برای سادگی، پیام را نگه می‌داریم (متن خالی است)
        }
        
        return $result;
    }
    
    /**
     * ارسال فایل
     * @param string $chatId شناسه چت
     * @param string $filePath مسیر فایل
     * @param string|null $caption کپشن فایل
     * @param string|null $customFilename نام سفارشی فایل (اختیاری)
     */
    public function sendDocument($chatId, $filePath, $caption = null, $customFilename = null) {
        if (!file_exists($filePath)) {
            error_log("File not found: $filePath");
            return ['error' => 'file_not_found'];
        }
        
        $realPath = realpath($filePath);
        $fileSize = filesize($realPath);
        $fileSizeMB = round($fileSize / (1024 * 1024), 2);
        
        // برای فایل‌های بزرگ‌تر از 50 مگابایت، تلاش می‌کنیم ارسال کنیم
        // اگر سرور Local Bot API داشته باشید، می‌توانید فایل‌هایی تا 2 گیگابایت ارسال کنید
        // در غیر این صورت، Telegram API خطای 413 خواهد داد
        
        $url = $this->apiUrl . 'sendDocument';
        
        $tempFile = null;
        
        // اگر نام سفارشی مشخص شده باشد، فایل را با نام جدید ارسال می‌کنیم
        if ($customFilename !== null && $customFilename !== '') {
            // پاک کردن کاراکترهای غیرمجاز از نام فایل
            $customFilename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $customFilename);
            $customFilename = trim($customFilename, '.');
            
            // اگر نام فایل خالی شد، از نام اصلی استفاده می‌کنیم
            if ($customFilename === '') {
                $customFilename = basename($realPath);
            }
            
            // اضافه کردن پسوند فایل اگر وجود نداشته باشد
            $originalExtension = pathinfo($realPath, PATHINFO_EXTENSION);
            if ($originalExtension && !preg_match('/\.' . preg_quote($originalExtension, '/') . '$/i', $customFilename)) {
                $customFilename .= '.' . $originalExtension;
            }
            
            // ایجاد یک فایل موقت با نام جدید
            // استفاده از یک مسیر موقت منحصر به فرد برای جلوگیری از تداخل
            $tempDir = sys_get_temp_dir();
            $uniqueId = uniqid('tg_', true);
            $tempFile = $tempDir . '/' . $uniqueId . '_' . $customFilename;
            
            if (!copy($realPath, $tempFile)) {
                error_log("Failed to create temp file: $tempFile");
                return ['error' => 'temp_file_failed'];
            }
            
            $fileToSend = $tempFile;
        } else {
            $fileToSend = $realPath;
        }
        
        // استفاده از CURLFile برای ارسال فایل
        $curlFile = new \CURLFile($fileToSend);
        
        // اگر نام سفارشی مشخص شده باشد، آن را در header تنظیم می‌کنیم
        // این کار نام فایل را در Telegram به عنوان نام سفارشی تنظیم می‌کند (بدون prefix)
        if ($customFilename !== null && $customFilename !== '' && method_exists($curlFile, 'setPostFilename')) {
            $curlFile->setPostFilename($customFilename);
        }
        
        $postData = [
            'chat_id' => $chatId,
            'document' => $curlFile
        ];
        
        if ($caption) {
            $postData['caption'] = $caption;
            $postData['parse_mode'] = 'HTML';
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // افزایش timeout برای فایل‌های بزرگ (بر اساس اندازه فایل)
        $timeout = 300; // حداقل 5 دقیقه
        if ($fileSize > 100 * 1024 * 1024) { // بیش از 100 مگابایت
            $timeout = 600; // 10 دقیقه
        }
        if ($fileSize > 500 * 1024 * 1024) { // بیش از 500 مگابایت
            $timeout = 1200; // 20 دقیقه
        }
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 60); // timeout برای اتصال
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        // حذف فایل موقت اگر ایجاد شده باشد
        if ($tempFile !== null && file_exists($tempFile)) {
            @unlink($tempFile);
        }
        
        if ($httpCode !== 200) {
            error_log("Telegram API error: HTTP $httpCode - $response");
            if ($httpCode === 413) {
                // اگر خطای 413 رخ داد، فایل خیلی بزرگ است
                return ['error' => 'file_too_large', 'size' => $fileSizeMB];
            }
            if ($curlError) {
                error_log("CURL error: $curlError");
                return ['error' => 'upload_failed', 'message' => $curlError];
            }
            return ['error' => 'api_error', 'code' => $httpCode];
        }
        
        $decoded = json_decode($response, true);
        if (!$decoded) {
            error_log("Failed to decode JSON response: $response");
            return ['error' => 'invalid_response'];
        }
        
        // بررسی پاسخ API برای خطا
        if (isset($decoded['ok']) && !$decoded['ok']) {
            error_log("Telegram API returned error: " . json_encode($decoded));
            return ['error' => 'api_error', 'message' => $decoded['description'] ?? 'Unknown error'];
        }
        
        return $decoded;
    }

    /**
     * دریافت وضعیت عضویت کاربر در کانال/گروه
     */
    public function getChatMember($chatId, $userId) {
        $data = [
            'chat_id' => $chatId,
            'user_id' => $userId
        ];

        return $this->sendRequest('getChatMember', $data);
    }
    
    /**
     * فوروارد پیام تلگرام
     */
    public function forwardMessage($chatId, $messageLink) {
        error_log("Forwarding message: " . $messageLink . " to chat: " . $chatId);
        
        // استخراج chat_id و message_id از لینک پیام
        $parsed = $this->parseTelegramMessageLink($messageLink);
        if (!$parsed) {
            error_log("Invalid Telegram message link: " . $messageLink);
            return false;
        }
        
        $data = [
            'chat_id' => $chatId,
            'from_chat_id' => $parsed['chat_id'],
            'message_id' => $parsed['message_id']
        ];
        
        error_log("Forward data: " . json_encode($data));
        
        $result = $this->sendRequest('forwardMessage', $data);
        
        if (!$result) {
            error_log("Failed to send forward request");
            return false;
        }
        
        if (!$result['ok']) {
            error_log("Telegram API error: " . json_encode($result));
            if (isset($result['description'])) {
                error_log("Error description: " . $result['description']);
            }
            return false;
        }
        
        error_log("Forward successful: " . json_encode($result));
        return $result;
    }
    
    /**
     * تجزیه لینک پیام تلگرام
     */
    private function parseTelegramMessageLink($link) {
        // حذف @ از ابتدای لینک
        $originalLink = $link;
        $link = ltrim($link, '@');
        
        error_log("Parsing link: " . $originalLink . " -> " . $link);
        
        // فرمت: https://t.me/username/123 یا https://t.me/c/chat_id/message_id
        $pattern = '/https:\/\/t\.me\/(?:c\/)?([a-zA-Z0-9_]+)\/(\d+)/';
        if (preg_match($pattern, $link, $matches)) {
            $chatId = $matches[1];
            $messageId = $matches[2];
            
            error_log("Pattern matched: chat_id=" . $chatId . ", message_id=" . $messageId);
            
            // اگر chat_id عددی است، مستقیماً استفاده می‌کنیم
            if (is_numeric($chatId)) {
                error_log("Numeric chat_id: " . $chatId);
                return [
                    'chat_id' => $chatId,
                    'message_id' => $messageId
                ];
            }
            
            // برای username ها، باید chat_id را از API دریافت کرد
            error_log("Username detected: " . $chatId . " - need to get chat_id from API");
            return false;
        }
        
        // بررسی فرمت‌های دیگر
        $pattern2 = '/https:\/\/t\.me\/([a-zA-Z0-9_]+)\/(\d+)/';
        if (preg_match($pattern2, $link, $matches)) {
            $username = $matches[1];
            $messageId = $matches[2];
            
            error_log("Pattern2 matched: username=" . $username . ", message_id=" . $messageId);
            return false;
        }
        
        error_log("No pattern matched for link: " . $link);
        return false;
    }
    
    /**
     * ویرایش پیام
     */
    public function editMessageText($chatId, $messageId, $text, $replyMarkup = null, $disableWebPagePreview = false) {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => $disableWebPagePreview
        ];
        
        if ($replyMarkup) {
            $data['reply_markup'] = $replyMarkup;
        }
        
        return $this->sendRequest('editMessageText', $data);
    }
    
    /**
     * حذف پیام
     */
    public function deleteMessage($chatId, $messageId) {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId
        ];
        
        return $this->sendRequest('deleteMessage', $data);
    }
    
    /**
     * تنظیم webhook
     */
    public function setWebhook($url) {
        $data = [
            'url' => $url
        ];
        
        return $this->sendRequest('setWebhook', $data);
    }
    
    /**
     * حذف webhook
     */
    public function deleteWebhook() {
        return $this->sendRequest('deleteWebhook');
    }
    
    /**
     * دریافت اطلاعات webhook
     */
    public function getWebhookInfo() {
        return $this->sendRequest('getWebhookInfo');
    }
    
    /**
     * دریافت اطلاعات بات
     */
    public function getMe() {
        return $this->sendRequest('getMe');
    }
    
    /**
     * پین کردن پیام
     */
    public function pinChatMessage($chatId, $messageId, $disableNotification = false) {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'disable_notification' => $disableNotification
        ];
        
        return $this->sendRequest('pinChatMessage', $data);
    }
    
    /**
     * آنپین کردن پیام
     */
    public function unpinChatMessage($chatId, $messageId = null) {
        $data = [
            'chat_id' => $chatId
        ];
        
        if ($messageId !== null) {
            $data['message_id'] = $messageId;
        }
        
        return $this->sendRequest('unpinChatMessage', $data);
    }
}
