<?php
require_once __DIR__ . '/BaseModel.php';

/**
 * مدیریت برچسب‌های محتوا
 */
class ContentTagModel extends BaseModel {
    protected $tableName = 'content_tags';

    /**
     * دریافت برچسب‌های یک آیتم
     */
    public function getTags($section, $itemId) {
        if (empty($section) || empty($itemId)) {
            return [];
        }

        $query = "SELECT tag FROM {$this->tableName} WHERE section = ? AND item_id = ? ORDER BY tag ASC";
        $rows = $this->db->select($query, [$section, $itemId]);
        return array_map(function($row) {
            return $row['tag'];
        }, $rows);
    }

    /**
     * دریافت برچسب‌های چند آیتم به صورت گروه‌بندی شده
     */
    public function getTagsForItems($section, array $itemIds) {
        $itemIds = array_filter(array_map('intval', $itemIds));
        if (empty($section) || empty($itemIds)) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($itemIds), '?'));
        $params = array_merge([$section], $itemIds);
        $query = "SELECT item_id, tag FROM {$this->tableName} WHERE section = ? AND item_id IN ({$placeholders}) ORDER BY tag ASC";

        $rows = $this->db->select($query, $params);
        $grouped = [];
        foreach ($rows as $row) {
            $id = (int)$row['item_id'];
            if (!isset($grouped[$id])) {
                $grouped[$id] = [];
            }
            $grouped[$id][] = $row['tag'];
        }
        return $grouped;
    }

    /**
     * تنظیم برچسب‌های یک آیتم (حذف قبلی و درج جدید)
     */
    public function setTags($section, $itemId, array $tags) {
        if (empty($section) || empty($itemId)) {
            return false;
        }

        $cleanTags = [];
        foreach ($tags as $tag) {
            $tag = trim($tag);
            if ($tag === '') {
                continue;
            }
            if (mb_strlen($tag) > 50) {
                $tag = mb_substr($tag, 0, 50);
            }
            if (!in_array($tag, $cleanTags, true)) {
                $cleanTags[] = $tag;
            }
            if (count($cleanTags) >= 10) {
                break;
            }
        }

        try {
            $this->db->beginTransaction();
            $this->db->delete("DELETE FROM {$this->tableName} WHERE section = ? AND item_id = ?", [$section, $itemId]);

            foreach ($cleanTags as $tag) {
                $this->db->insert(
                    "INSERT INTO {$this->tableName} (section, item_id, tag) VALUES (?, ?, ?)",
                    [$section, $itemId, $tag]
                );
            }

            $this->db->commit();
            return $cleanTags;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("ContentTagModel::setTags error: " . $e->getMessage());
            return false;
        }
    }
}


