<?php
require_once __DIR__ . '/BaseModel.php';

/**
 * کلاس مدیریت کاربران بات تلگرام
 */
class BotUserModel extends BaseModel {
    protected $tableName = 'bot_users';
    
    /**
     * ثبت یا به‌روزرسانی کاربر
     * اگر کاربر وجود داشته باشد، اطلاعات را به‌روزرسانی می‌کند
     * اگر وجود نداشته باشد، کاربر جدید را ثبت می‌کند
     * @return int|false شناسه کاربر در صورت موفقیت، false در صورت خطا
     */
    public function registerOrUpdateUser($chatId, $userInfo) {
        // بررسی وجود کاربر
        $existingUser = $this->getByChatId($chatId);
        
        if ($existingUser) {
            // به‌روزرسانی اطلاعات کاربر (بدون افزایش activity_count - این کار در incrementActivity انجام می‌شود)
            $updateData = [
                'username' => !empty($userInfo['username']) ? $userInfo['username'] : $existingUser['username'],
                'first_name' => !empty($userInfo['first_name']) ? $userInfo['first_name'] : $existingUser['first_name'],
                'last_name' => !empty($userInfo['last_name']) ? $userInfo['last_name'] : $existingUser['last_name'],
                'last_seen_at' => date('Y-m-d H:i:s')
                // activity_count در incrementActivity افزایش می‌یابد
            ];
            
            $result = $this->updateByChatId($chatId, $updateData);
            // اگر به‌روزرسانی موفق بود، ID کاربر را برمی‌گردانیم
            return $result ? ($existingUser['id'] ?? false) : false;
        } else {
            // ثبت کاربر جدید
            $data = [
                'chat_id' => $chatId,
                'username' => $userInfo['username'] ?? null,
                'first_name' => $userInfo['first_name'] ?? null,
                'last_name' => $userInfo['last_name'] ?? null,
                'activity_count' => 1,
                'first_seen_at' => date('Y-m-d H:i:s'),
                'last_seen_at' => date('Y-m-d H:i:s')
            ];
            
            return $this->create($data);
        }
    }
    
    /**
     * دریافت کاربر بر اساس chat_id
     */
    public function getByChatId($chatId) {
        $query = "SELECT * FROM {$this->tableName} WHERE chat_id = ?";
        return $this->db->selectOne($query, [$chatId]);
    }
    
    /**
     * به‌روزرسانی کاربر بر اساس chat_id
     */
    public function updateByChatId($chatId, $data) {
        $setParts = [];
        $params = [];
        
        foreach ($data as $key => $value) {
            $setParts[] = "{$key} = ?";
            $params[] = $value;
        }
        
        $params[] = $chatId;
        $setClause = implode(', ', $setParts);
        
        $query = "UPDATE {$this->tableName} SET {$setClause} WHERE chat_id = ?";
        return $this->db->update($query, $params);
    }
    
    /**
     * افزایش تعداد فعالیت کاربر
     */
    public function incrementActivity($chatId) {
        $query = "UPDATE {$this->tableName} 
                  SET activity_count = activity_count + 1, 
                      last_seen_at = NOW() 
                  WHERE chat_id = ?";
        return $this->db->update($query, [$chatId]);
    }
    
    /**
     * دریافت تمام کاربران با ترتیب فعالیت
     */
    public function getAllUsers($limit = null, $offset = 0, $orderBy = 'activity_count DESC') {
        $query = "SELECT * FROM {$this->tableName} ORDER BY {$orderBy}";
        
        if ($limit !== null) {
            $query .= " LIMIT {$limit}";
            if ($offset > 0) {
                $query .= " OFFSET {$offset}";
            }
        }
        
        return $this->db->select($query);
    }
    
    /**
     * شمارش کل کاربران
     */
    public function getTotalUsers() {
        $query = "SELECT COUNT(*) as total FROM {$this->tableName}";
        $result = $this->db->selectOne($query);
        return $result['total'] ?? 0;
    }
    
    /**
     * دریافت کاربران فعال (با فعالیت در بازه زمانی مشخص)
     */
    public function getActiveUsers($days = 7, $limit = 20) {
        $query = "SELECT * FROM {$this->tableName} 
                  WHERE last_seen_at >= DATE_SUB(NOW(), INTERVAL ? DAY) 
                  ORDER BY activity_count DESC 
                  LIMIT ?";
        
        return $this->db->select($query, [$days, $limit]);
    }
    
    /**
     * دریافت آمار کاربران
     */
    public function getUserStats() {
        $query = "SELECT 
                    COUNT(*) AS total_users,
                    SUM(activity_count) AS total_activities,
                    AVG(activity_count) AS avg_activities_per_user,
                    COUNT(CASE WHEN last_seen_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 END) AS active_users_7_days,
                    COUNT(CASE WHEN last_seen_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) AS active_users_30_days,
                    COUNT(CASE WHEN DATE(last_seen_at) = CURDATE() THEN 1 END) AS active_users_today
                  FROM {$this->tableName}";
        
        $result = $this->db->selectOne($query);
        
        return $result ?: [
            'total_users' => 0,
            'total_activities' => 0,
            'avg_activities_per_user' => 0,
            'active_users_7_days' => 0,
            'active_users_30_days' => 0,
            'active_users_today' => 0
        ];
    }
    
    /**
     * ایجاد کاربر جدید
     */
    public function create($data) {
        $query = "INSERT INTO {$this->tableName} 
                  (chat_id, username, first_name, last_name, activity_count, first_seen_at, last_seen_at) 
                  VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $data['chat_id'],
            $data['username'] ?? null,
            $data['first_name'] ?? null,
            $data['last_name'] ?? null,
            $data['activity_count'] ?? 1,
            $data['first_seen_at'] ?? date('Y-m-d H:i:s'),
            $data['last_seen_at'] ?? date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert($query, $params);
    }
}

