<?php
require_once __DIR__ . '/../Database/Database.php';

/**
 * کلاس پایه برای مدل‌ها
 */
abstract class BaseModel {
    protected $db;
    protected $tableName;
    protected $sectionKey = null;
    protected static $columnCache = [];
    
    public function __construct() {
        $this->db = new Database();
    }
    
    /**
     * بررسی وجود یک ستون در جدول
     */
    public function columnExists($columnName) {
        if (empty($this->tableName) || empty($columnName)) {
            return false;
        }
        
        $cacheKey = $this->tableName . '_' . $columnName;
        if (isset(self::$columnCache[$cacheKey])) {
            return self::$columnCache[$cacheKey];
        }
        
        try {
            // استفاده از INFORMATION_SCHEMA که برای PDO بهتر است
            require_once __DIR__ . '/../../config/database.php';
            $dbName = DatabaseConfig::DB_NAME;
            
            if (empty($dbName)) {
                error_log("BaseModel::columnExists: DB_NAME is empty");
                return false;
            }
            
            $query = "SELECT COUNT(*) as cnt FROM INFORMATION_SCHEMA.COLUMNS 
                      WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? AND COLUMN_NAME = ?";
            $result = $this->db->selectOne($query, [$dbName, $this->tableName, $columnName]);
            $exists = !empty($result) && isset($result['cnt']) && $result['cnt'] > 0;
            self::$columnCache[$cacheKey] = $exists;
            return $exists;
        } catch (Exception $e) {
            error_log("ColumnExists error for {$this->tableName}.{$columnName}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * دریافت رکورد با اطلاعات سازنده
     */
    public function getByIdWithCreator($id) {
        // بررسی اعتبار ID
        if (empty($id) || !is_numeric($id) || $id <= 0) {
            return false;
        }
        
        if (empty($this->tableName)) {
            error_log("BaseModel::getByIdWithCreator: tableName is empty");
            return false;
        }
        
        $query = "SELECT t.*, u.username as creator_username, u.full_name as creator_name 
                  FROM {$this->tableName} t 
                  LEFT JOIN admin_users u ON t.created_by = u.id 
                  WHERE t.id = ?";
        $result = $this->db->selectOne($query, [(int)$id]);
        
        // نرمال‌سازی parent_id: تبدیل 0 به null
        if ($result && is_array($result) && isset($result['parent_id']) && ($result['parent_id'] === 0 || $result['parent_id'] === '0')) {
            $result['parent_id'] = null;
        }
        
        return $result;
    }
    
    /**
     * دریافت تمام رکوردها با اطلاعات سازنده
     */
    public function getAllWithCreator($limit = null, $offset = 0) {
        $orderBy = "ORDER BY t.created_at DESC";
        if ($this->columnExists('parent_id') && $this->columnExists('display_order')) {
            $orderBy = "ORDER BY t.parent_id ASC, t.display_order ASC, t.created_at DESC";
        } elseif ($this->columnExists('display_order')) {
            $orderBy = "ORDER BY t.display_order ASC, t.created_at DESC";
        }
        
        $query = "SELECT t.*, u.username as creator_username, u.full_name as creator_name 
                  FROM {$this->tableName} t 
                  LEFT JOIN admin_users u ON t.created_by = u.id 
                  {$orderBy}";
        
        if ($limit) {
            $query .= " LIMIT {$limit}";
            if ($offset > 0) {
                $query .= " OFFSET {$offset}";
            }
        }
        
        $results = $this->db->select($query);
        
        // نرمال‌سازی parent_id: تبدیل 0 به null
        foreach ($results as &$result) {
            if (isset($result['parent_id']) && ($result['parent_id'] === 0 || $result['parent_id'] === '0')) {
                $result['parent_id'] = null;
            }
        }
        
        return $results;
    }
    
    /**
     * دریافت مطالب یک ادمین خاص
     */
    public function getByCreator($creatorId) {
        $query = "SELECT * FROM {$this->tableName} WHERE created_by = ? ORDER BY created_at DESC";
        return $this->db->select($query, [$creatorId]);
    }
    
    /**
     * دریافت تمام رکوردهای فعال
     */
    public function getAll($limit = null, $offset = 0) {
        $orderBy = "ORDER BY created_at DESC";
        $hasPinned = $this->columnExists('is_pinned');
        
        if ($this->columnExists('parent_id') && $this->columnExists('display_order')) {
            if ($hasPinned) {
                $orderBy = "ORDER BY is_pinned DESC, parent_id ASC, display_order ASC, created_at DESC";
            } else {
                $orderBy = "ORDER BY parent_id ASC, display_order ASC, created_at DESC";
            }
        } elseif ($this->columnExists('display_order')) {
            if ($hasPinned) {
                $orderBy = "ORDER BY is_pinned DESC, display_order ASC, created_at DESC";
            } else {
                $orderBy = "ORDER BY display_order ASC, created_at DESC";
            }
        } elseif ($hasPinned) {
            $orderBy = "ORDER BY is_pinned DESC, created_at DESC";
        }
        
        $query = "SELECT * FROM {$this->tableName} {$orderBy}";
        
        if ($limit) {
            $query .= " LIMIT {$limit}";
            if ($offset > 0) {
                $query .= " OFFSET {$offset}";
            }
        }
        
        $results = $this->db->select($query);
        
        // نرمال‌سازی parent_id: تبدیل 0 به null
        foreach ($results as &$result) {
            if (isset($result['parent_id']) && ($result['parent_id'] === 0 || $result['parent_id'] === '0')) {
                $result['parent_id'] = null;
            }
        }
        
        return $results;
    }
    
    /**
     * دریافت رکورد بر اساس ID
     */
    public function getById($id) {
        // بررسی اعتبار ID
        if (empty($id) || !is_numeric($id) || $id <= 0) {
            return false;
        }
        
        if (empty($this->tableName)) {
            error_log("BaseModel::getById: tableName is empty");
            return false;
        }
        
        $query = "SELECT * FROM {$this->tableName} WHERE id = ?";
        $result = $this->db->selectOne($query, [(int)$id]);
        
        // نرمال‌سازی parent_id: تبدیل 0 به null
        if ($result && is_array($result) && isset($result['parent_id']) && ($result['parent_id'] === 0 || $result['parent_id'] === '0')) {
            $result['parent_id'] = null;
        }
        
        return $result;
    }
    
    /**
     * جستجو در رکوردها (پیشرفته - جستجوی چند کلمه‌ای)
     */
    public function search($keyword, $limit = null, $offset = 0) {
        // تقسیم کلمات جستجو
        $keywords = preg_split('/\s+/u', trim($keyword));
        $keywords = array_filter($keywords, function($k) {
            return mb_strlen(trim($k)) > 0;
        });
        
        if (empty($keywords)) {
            return [];
        }
        
        // ساخت شرط‌های جستجو برای هر کلمه
        $titleConditions = [];
        $descriptionConditions = [];
        $tagConditions = [];
        $params = [];
        
        if (!empty($this->sectionKey)) {
            $params[] = $this->sectionKey;
        }
        
        foreach ($keywords as $kw) {
            $kwTerm = "%{$kw}%";
            $titleConditions[] = "t.title LIKE ?";
            $params[] = $kwTerm;
            
            // اگر description وجود دارد، در آن هم جستجو می‌کنیم
            if ($this->columnExists('description')) {
                $descriptionConditions[] = "t.description LIKE ?";
                $params[] = $kwTerm;
            }
            
            // جستجو در تگ‌ها
            if (!empty($this->sectionKey)) {
                $tagConditions[] = "ct.tag LIKE ?";
                $params[] = $kwTerm;
            }
        }
        
        // ساخت query
        $whereParts = [];
        
        // شرط title (حداقل یکی از کلمات باید در title باشد)
        if (!empty($titleConditions)) {
            $whereParts[] = "(" . implode(" OR ", $titleConditions) . ")";
        }
        
        // شرط description (اختیاری - اگر description وجود دارد)
        if (!empty($descriptionConditions)) {
            $whereParts[] = "(" . implode(" OR ", $descriptionConditions) . ")";
        }
        
        // شرط tags
        if (!empty($tagConditions) && !empty($this->sectionKey)) {
            $whereParts[] = "(" . implode(" OR ", $tagConditions) . ")";
        }
        
        if (empty($whereParts)) {
            return [];
        }
        
        $whereClause = "(" . implode(" OR ", $whereParts) . ")";
        
        // ساخت query نهایی
        if (!empty($this->sectionKey)) {
            $query = "SELECT DISTINCT t.* FROM {$this->tableName} t 
                      LEFT JOIN content_tags ct ON ct.section = ? AND ct.item_id = t.id
                      WHERE {$whereClause}
                      ORDER BY 
                        CASE 
                            WHEN t.title LIKE ? THEN 1
                            WHEN " . ($this->columnExists('description') ? "t.description LIKE ?" : "0=1") . " THEN 2
                            ELSE 3
                        END,
                        t.created_at DESC";
            // اضافه کردن پارامتر برای ORDER BY
            $params[] = "%{$keywords[0]}%";
            if ($this->columnExists('description')) {
                $params[] = "%{$keywords[0]}%";
            }
        } else {
            $query = "SELECT * FROM {$this->tableName} 
                      WHERE {$whereClause}
                      ORDER BY 
                        CASE 
                            WHEN title LIKE ? THEN 1
                            WHEN " . ($this->columnExists('description') ? "description LIKE ?" : "0=1") . " THEN 2
                            ELSE 3
                        END,
                        created_at DESC";
            $params[] = "%{$keywords[0]}%";
            if ($this->columnExists('description')) {
                $params[] = "%{$keywords[0]}%";
            }
        }

        if ($limit) {
            $query .= " LIMIT {$limit}";
            if ($offset > 0) {
                $query .= " OFFSET {$offset}";
            }
        }
        
        return $this->db->select($query, $params);
    }
    
    /**
     * شمارش کل رکوردهای فعال
     */
    public function countAll() {
        if (empty($this->tableName)) {
            error_log("BaseModel::countAll: tableName is empty");
            return 0;
        }
        $query = "SELECT COUNT(*) as total FROM {$this->tableName}";
        $result = $this->db->selectOne($query);
        return $result && isset($result['total']) ? (int)$result['total'] : 0;
    }
    
    /**
     * شمارش نتایج جستجو
     */
    public function countSearch($keyword) {
        if (empty($this->tableName)) {
            error_log("BaseModel::countSearch: tableName is empty");
            return 0;
        }
        $searchTerm = "%{$keyword}%";
        if (!empty($this->sectionKey)) {
            $query = "SELECT COUNT(DISTINCT t.id) as total FROM {$this->tableName} t 
                      LEFT JOIN content_tags ct ON ct.section = ? AND ct.item_id = t.id
                      WHERE t.title LIKE ? OR ct.tag LIKE ?";
            $params = [$this->sectionKey, $searchTerm, $searchTerm];
        } else {
            $query = "SELECT COUNT(*) as total FROM {$this->tableName} WHERE title LIKE ?";
            $params = [$searchTerm];
        }
        $result = $this->db->selectOne($query, $params);
        return $result && isset($result['total']) ? (int)$result['total'] : 0;
    }
    
    /**
     * ایجاد رکورد جدید
     */
    public function create($data) {
        // فیلتر کردن فیلدهایی که وجود ندارند (فقط برای فیلدهای جدید)
        $filteredData = [];
        $newFields = ['file_path', 'parent_id', 'display_order', 'item_type']; // فیلدهای جدید که ممکن است وجود نداشته باشند
        
        foreach ($data as $key => $value) {
            // اگر فیلد جدید است، بررسی می‌کنیم وجود دارد یا نه
            if (in_array($key, $newFields)) {
                if ($this->columnExists($key)) {
                    $filteredData[$key] = $value;
                }
            } else {
                // برای فیلدهای قدیمی، مستقیماً اضافه می‌کنیم
                $filteredData[$key] = $value;
            }
        }
        
        // اگر هیچ فیلدی برای درج باقی نماند، false برمی‌گردانیم
        if (empty($filteredData)) {
            return false;
        }
        
        $fields = array_keys($filteredData);
        $placeholders = array_fill(0, count($fields), '?');
        
        // تبدیل parent_id = 0 به null برای MySQL
        $params = array_values($filteredData);
        foreach ($params as $key => $value) {
            if ($fields[$key] === 'parent_id' && ($value === 0 || $value === '0')) {
                $params[$key] = null;
            }
        }
        
        $query = "INSERT INTO {$this->tableName} (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $placeholders) . ")";
        return $this->db->insert($query, $params);
    }
    
    /**
     * به‌روزرسانی رکورد
     */
    public function update($id, $data) {
        // بررسی اعتبار ID
        if (empty($id) || !is_numeric($id) || $id <= 0) {
            error_log("BaseModel::update: Invalid ID: " . $id);
            return false;
        }
        
        if (empty($this->tableName)) {
            error_log("BaseModel::update: tableName is empty");
            return false;
        }
        
        // فیلتر کردن فیلدهایی که وجود ندارند (فقط برای فیلدهای جدید)
        $filteredData = [];
        $newFields = ['file_path', 'parent_id', 'display_order', 'item_type', 'is_pinned']; // فیلدهای جدید که ممکن است وجود نداشته باشند
        
        foreach ($data as $key => $value) {
            // اگر فیلد جدید است، بررسی می‌کنیم وجود دارد یا نه
            if (in_array($key, $newFields)) {
                if ($this->columnExists($key)) {
                    $filteredData[$key] = $value;
                } else {
                    // اگر ستون وجود ندارد، لاگ می‌کنیم
                    error_log("BaseModel::update: Column '{$key}' does not exist in table '{$this->tableName}' - skipping update");
                    // برای فیلدهای مهم مثل display_order و is_pinned، اگر ستون وجود نداشته باشد، فیلتر می‌کنیم
                    // اما برای سایر فیلدها، می‌توانیم اضافه کنیم
                    if ($key === 'display_order') {
                        // display_order باید وجود داشته باشد، پس اگر وجود ندارد، اضافه نمی‌کنیم
                        error_log("BaseModel::update: Skipping '{$key}' update - column does not exist");
                    } elseif ($key === 'is_pinned') {
                        // is_pinned باید وجود داشته باشد، پس اگر وجود ندارد، اضافه نمی‌کنیم
                        error_log("BaseModel::update: Skipping '{$key}' update - column does not exist in table '{$this->tableName}'");
                    }
                }
            } else {
                // برای فیلدهای قدیمی، مستقیماً اضافه می‌کنیم
                $filteredData[$key] = $value;
            }
        }
        
        // اگر هیچ فیلدی برای بروزرسانی باقی نماند، false برمی‌گردانیم
        if (empty($filteredData)) {
            error_log("BaseModel::update: No fields to update for table '{$this->tableName}', ID: {$id}");
            return false;
        }
        
        $fields = array_keys($filteredData);
        
        if (empty($fields)) {
            error_log("BaseModel::update: No fields to update after filtering");
            return false;
        }
        
        $setClause = implode(' = ?, ', $fields) . ' = ?';
        
        $query = "UPDATE {$this->tableName} SET {$setClause} WHERE id = ?";
        $params = array_values($filteredData);
        
        // تبدیل parent_id = 0 به null برای MySQL
        foreach ($params as $key => $value) {
            if ($fields[$key] === 'parent_id' && ($value === 0 || $value === '0')) {
                $params[$key] = null;
            }
        }
        
        $params[] = (int)$id;
        
        // لاگ برای دیباگ
        error_log("BaseModel::update - Table: {$this->tableName}, ID: {$id}, Fields: " . implode(', ', $fields));
        
        return $this->db->update($query, $params);
    }
    
    /**
     * حذف واقعی رکورد
     */
    public function delete($id) {
        $query = "DELETE FROM {$this->tableName} WHERE id = ?";
        return $this->db->delete($query, [$id]);
    }
    
    /**
     * دریافت مطالب با پیجینیشن
     */
    public function getPaginated($page, $limit) {
        $offset = ($page - 1) * $limit;
        $orderBy = "ORDER BY created_at DESC";
        $hasPinned = $this->columnExists('is_pinned');
        
        if ($this->columnExists('parent_id') && $this->columnExists('display_order')) {
            if ($hasPinned) {
                $orderBy = "ORDER BY is_pinned DESC, parent_id ASC, display_order ASC, created_at DESC";
            } else {
                $orderBy = "ORDER BY parent_id ASC, display_order ASC, created_at DESC";
            }
        } elseif ($this->columnExists('display_order')) {
            if ($hasPinned) {
                $orderBy = "ORDER BY is_pinned DESC, display_order ASC, created_at DESC";
            } else {
                $orderBy = "ORDER BY display_order ASC, created_at DESC";
            }
        } elseif ($hasPinned) {
            $orderBy = "ORDER BY is_pinned DESC, created_at DESC";
        }
        
        $query = "SELECT * FROM {$this->tableName} {$orderBy} LIMIT ? OFFSET ?";
        return $this->db->select($query, [$limit, $offset]);
    }
    
    /**
     * محاسبه تعداد کل صفحات
     */
    public function getTotalPages($limit) {
        $total = $this->countAll();
        return ceil($total / $limit);
    }
    
    /**
     * اضافه کردن ستون is_pinned به جدول در صورت عدم وجود
     */
    public function addIsPinnedColumnIfNotExists() {
        if (empty($this->tableName)) {
            return false;
        }
        
        // بررسی وجود ستون
        if ($this->columnExists('is_pinned')) {
            return true;
        }
        
        try {
            // بررسی وجود ستون created_at برای تعیین موقعیت
            $hasCreatedAt = $this->columnExists('created_at');
            $position = $hasCreatedAt ? "AFTER `created_at`" : "";
            
            // اضافه کردن ستون is_pinned
            $query = "ALTER TABLE `{$this->tableName}` ADD COLUMN `is_pinned` TINYINT(1) NOT NULL DEFAULT 0 {$position}";
            $result = $this->db->execute($query);
            
            if ($result !== false) {
                // پاک کردن کش
                $cacheKey = $this->tableName . '_is_pinned';
                unset(self::$columnCache[$cacheKey]);
                error_log("Added is_pinned column to table: {$this->tableName}");
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Error adding is_pinned column to {$this->tableName}: " . $e->getMessage());
            return false;
        }
    }
}