<?php
require_once __DIR__ . '/BaseModel.php';

/**
 * کلاس مدیریت ادمین‌ها
 */
class AdminModel extends BaseModel {
    protected $tableName = 'admin_users';
    
    /**
     * ایجاد ادمین جدید
     */
    public function create($data) {
        $password = password_hash($data['password'], PASSWORD_DEFAULT);
        
        $query = "INSERT INTO {$this->tableName} (username, password, full_name, role, created_by) VALUES (?, ?, ?, ?, ?)";
        $params = [
            $data['username'],
            $password,
            $data['full_name'],
            $data['role'] ?? 'admin',
            $data['created_by'] ?? null
        ];
        
        return $this->db->insert($query, $params);
    }
    
    /**
     * بررسی نام کاربری و رمز عبور
     */
    public function authenticate($username, $password) {
        $query = "SELECT * FROM {$this->tableName} WHERE username = ?";
        $user = $this->db->selectOne($query, [$username]);
        
        if ($user && password_verify($password, $user['password'])) {
            // به‌روزرسانی زمان آخرین ورود
            $this->updateLastLogin($user['id']);
            return $user;
        }
        
        return null;
    }
    
    /**
     * به‌روزرسانی زمان آخرین ورود (با timezone ایران)
     */
    public function updateLastLogin($id) {
        // تنظیم timezone به ایران
        date_default_timezone_set('Asia/Tehran');
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->tableName} SET last_login = ? WHERE id = ?";
        return $this->db->update($query, [$now, $id]);
    }
    
    /**
     * دریافت ادمین بر اساس ID (بدون شرط is_active)
     */
    public function getById($id) {
        $query = "SELECT * FROM {$this->tableName} WHERE id = ?";
        return $this->db->selectOne($query, [$id]);
    }
    
    /**
     * بررسی دسترسی برای ویرایش/حذف مطلب
     */
    public function canModifyContent($adminId, $contentCreatorId, $adminRole) {
        // همه ادمین‌ها (سوپر ادمین و ادمین عادی) می‌توانند همه محتواها را ویرایش/حذف کنند
        return true;
    }
    
    /**
     * بررسی دسترسی مدیریت ادمین‌ها
     */
    public function canManageAdmins($adminRole) {
        return $adminRole === 'super_admin';
    }
    
    /**
     * دریافت تمام ادمین‌ها برای سوپر ادمین
     */
    public function getAllAdmins() {
        $query = "SELECT id, username, full_name, role, created_at, last_login FROM {$this->tableName} ORDER BY created_at DESC";
        return $this->db->select($query);
    }
    
    /**
     * تغییر نقش ادمین
     */
    public function changeRole($id, $role) {
        $query = "UPDATE {$this->tableName} SET role = ? WHERE id = ?";
        return $this->db->update($query, [$role, $id]);
    }
    
    /**
     * تغییر رمز عبور
     */
    public function changePassword($id, $newPassword) {
        $password = password_hash($newPassword, PASSWORD_DEFAULT);
        $query = "UPDATE {$this->tableName} SET password = ? WHERE id = ?";
        return $this->db->update($query, [$password, $id]);
    }
    
    /**
     * دریافت ادمین بر اساس نام کاربری
     */
    public function getByUsername($username) {
        $query = "SELECT * FROM {$this->tableName} WHERE username = ?";
        return $this->db->selectOne($query, [$username]);
    }
    
    /**
     * به‌روزرسانی اطلاعات ادمین
     */
    public function update($id, $data) {
        $query = "UPDATE {$this->tableName} SET username = ?, role = ? WHERE id = ?";
        $params = [
            $data['username'],
            $data['role'],
            $id
        ];
        
        return $this->db->update($query, $params);
    }
    
    /**
     * حذف ادمین
     */
    public function delete($id) {
        $query = "DELETE FROM {$this->tableName} WHERE id = ?";
        return $this->db->delete($query, [$id]);
    }
    
    /**
     * حذف ادمین (فقط سوپر ادمین)
     */
    public function deleteAdmin($id) {
        // بررسی که خودش را حذف نکند
        $query = "DELETE FROM {$this->tableName} WHERE id = ? AND role != 'super_admin'";
        return $this->db->delete($query, [$id]);
    }
}