<?php
require_once __DIR__ . '/BaseModel.php';

/**
 * کلاس مدیریت لاگ فعالیت‌های کاربران
 */
class ActivityLogModel extends BaseModel {
    protected $tableName = 'bot_activity_logs';
    
    /**
     * ثبت لاگ فعالیت
     */
    public function logActivity($data) {
        // بررسی وجود ستون user_id
        $hasUserId = $this->columnExists('user_id');
        
        if ($hasUserId) {
            $query = "INSERT INTO {$this->tableName} 
                      (chat_id, user_id, username, first_name, last_name, action_type, action_name, section, item_id, item_title, metadata) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $params = [
                $data['chat_id'] ?? null,
                $data['user_id'] ?? null,
                $data['username'] ?? null,
                $data['first_name'] ?? null,
                $data['last_name'] ?? null,
                $data['action_type'] ?? 'unknown',
                $data['action_name'] ?? null,
                $data['section'] ?? null,
                $data['item_id'] ?? null,
                $data['item_title'] ?? null,
                isset($data['metadata']) ? json_encode($data['metadata']) : null
            ];
        } else {
            // اگر ستون user_id وجود ندارد، از کوئری قدیمی استفاده می‌کنیم
            $query = "INSERT INTO {$this->tableName} 
                      (chat_id, username, first_name, last_name, action_type, action_name, section, item_id, item_title, metadata) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $params = [
                $data['chat_id'] ?? null,
                $data['username'] ?? null,
                $data['first_name'] ?? null,
                $data['last_name'] ?? null,
                $data['action_type'] ?? 'unknown',
                $data['action_name'] ?? null,
                $data['section'] ?? null,
                $data['item_id'] ?? null,
                $data['item_title'] ?? null,
                isset($data['metadata']) ? json_encode($data['metadata']) : null
            ];
        }
        
        return $this->db->insert($query, $params);
    }
    
    /**
     * دریافت آمار کلیک روی دکمه‌ها
     */
    public function getButtonClickStats($limit = 10) {
        $query = "SELECT action_name, COUNT(*) as click_count 
                  FROM {$this->tableName} 
                  WHERE action_type = 'button_click' 
                  GROUP BY action_name 
                  ORDER BY click_count DESC 
                  LIMIT ?";
        
        return $this->db->select($query, [$limit]);
    }
    
    /**
     * دریافت آمار استفاده از بخش‌ها
     */
    public function getSectionStats($limit = 10) {
        $query = "SELECT section, COUNT(*) as usage_count 
                  FROM {$this->tableName} 
                  WHERE section IS NOT NULL 
                  GROUP BY section 
                  ORDER BY usage_count DESC 
                  LIMIT ?";
        
        return $this->db->select($query, [$limit]);
    }
    
    /**
     * دریافت آمار مشاهده فایل‌ها
     */
    public function getFileViewStats($limit = 20) {
        $query = "SELECT item_title, section, COUNT(*) as view_count 
                  FROM {$this->tableName} 
                  WHERE action_type = 'file_view' AND item_title IS NOT NULL 
                  GROUP BY item_title, section 
                  ORDER BY view_count DESC 
                  LIMIT ?";
        
        return $this->db->select($query, [$limit]);
    }
    
    /**
     * دریافت آمار کاربران فعال (از جدول bot_users)
     * این متد دیگر استفاده نمی‌شود و به جای آن از BotUserModel استفاده می‌شود
     */
    public function getActiveUsersStats($limit = 20) {
        // استفاده از جدول bot_users به جای گروه‌بندی از activity_logs
        $hasUsersTable = false;
        try {
            $checkQuery = "SELECT COUNT(*) as cnt FROM information_schema.tables 
                          WHERE table_schema = DATABASE() AND table_name = 'bot_users'";
            $result = $this->db->selectOne($checkQuery);
            $hasUsersTable = !empty($result) && isset($result['cnt']) && $result['cnt'] > 0;
        } catch (Exception $e) {
            error_log("Error checking bot_users table: " . $e->getMessage());
        }
        
        if ($hasUsersTable) {
            $query = "SELECT 
                        id,
                        chat_id, 
                        username, 
                        first_name, 
                        last_name,
                        activity_count,
                        last_seen_at as last_activity
                      FROM bot_users 
                      ORDER BY activity_count DESC, last_seen_at DESC 
                      LIMIT ?";
            return $this->db->select($query, [$limit]);
        } else {
            // اگر جدول bot_users وجود ندارد، از روش قدیمی استفاده می‌کنیم
            $query = "SELECT 
                        chat_id, 
                        username, 
                        first_name, 
                        last_name,
                        COUNT(*) as activity_count,
                        MAX(created_at) as last_activity
                      FROM {$this->tableName} 
                      GROUP BY chat_id, username, first_name, last_name 
                      ORDER BY activity_count DESC 
                      LIMIT ?";
            
            return $this->db->select($query, [$limit]);
        }
    }
    
    /**
     * دریافت آمار فعالیت روزانه
     */
    public function getDailyStats($days = 30) {
        $query = "SELECT 
                    DATE(created_at) as date,
                    COUNT(*) as total_activities,
                    COUNT(DISTINCT chat_id) as unique_users
                  FROM {$this->tableName} 
                  WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                  GROUP BY DATE(created_at)
                  ORDER BY date DESC";
        
        return $this->db->select($query, [$days]);
    }
    
    /**
     * دریافت آمار فعالیت ساعتی
     */
    public function getHourlyStats($days = 7) {
        $query = "SELECT 
                    HOUR(created_at) as hour,
                    COUNT(*) as activity_count
                  FROM {$this->tableName} 
                  WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                  GROUP BY HOUR(created_at)
                  ORDER BY hour";
        
        return $this->db->select($query, [$days]);
    }
    
    /**
     * دریافت کل لاگ‌ها با فیلتر
     */
    public function getAllLogs($limit = 100, $offset = 0, $filters = []) {
        $where = [];
        $params = [];
        
        if (!empty($filters['chat_id'])) {
            $where[] = "chat_id = ?";
            $params[] = $filters['chat_id'];
        }
        
        if (!empty($filters['action_type'])) {
            $where[] = "action_type = ?";
            $params[] = $filters['action_type'];
        }
        
        if (!empty($filters['section'])) {
            $where[] = "section = ?";
            $params[] = $filters['section'];
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
        
        $query = "SELECT * FROM {$this->tableName} {$whereClause} ORDER BY created_at DESC LIMIT ? OFFSET ?";
        
        $params[] = $limit;
        $params[] = $offset;
        
        return $this->db->select($query, $params);
    }
    
    /**
     * شمارش کل لاگ‌ها
     */
    public function getTotalLogs($filters = []) {
        $where = [];
        $params = [];
        
        if (!empty($filters['chat_id'])) {
            $where[] = "chat_id = ?";
            $params[] = $filters['chat_id'];
        }
        
        if (!empty($filters['action_type'])) {
            $where[] = "action_type = ?";
            $params[] = $filters['action_type'];
        }
        
        if (!empty($filters['section'])) {
            $where[] = "section = ?";
            $params[] = $filters['section'];
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
        
        $query = "SELECT COUNT(*) as total FROM {$this->tableName} {$whereClause}";
        $result = $this->db->selectOne($query, $params);
        
        return $result['total'] ?? 0;
    }

    /**
     * دریافت آمار خلاصه
     */
    public function getSummaryStats() {
        // بررسی وجود جدول bot_users
        $hasUsersTable = false;
        try {
            $checkQuery = "SELECT COUNT(*) as cnt FROM information_schema.tables 
                          WHERE table_schema = DATABASE() AND table_name = 'bot_users'";
            $result = $this->db->selectOne($checkQuery);
            $hasUsersTable = !empty($result) && isset($result['cnt']) && $result['cnt'] > 0;
        } catch (Exception $e) {
            error_log("Error checking bot_users table: " . $e->getMessage());
        }
        
        if ($hasUsersTable) {
            // استفاده از جدول bot_users برای تعداد کاربران
            $query = "SELECT 
                        (SELECT COUNT(*) FROM {$this->tableName}) AS total_activities,
                        (SELECT COUNT(*) FROM bot_users) AS total_users,
                        SUM(CASE WHEN action_type = 'button_click' THEN 1 ELSE 0 END) AS total_button_clicks,
                        SUM(CASE WHEN action_type = 'section_view' THEN 1 ELSE 0 END) AS total_section_views,
                        SUM(CASE WHEN action_type = 'file_view' THEN 1 ELSE 0 END) AS total_file_views,
                        SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) AS activities_last_7_days,
                        (SELECT COUNT(*) FROM bot_users WHERE last_seen_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)) AS active_users_last_7_days,
                        SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) AS activities_today,
                        (SELECT COUNT(*) FROM bot_users WHERE DATE(last_seen_at) = CURDATE()) AS active_users_today
                      FROM {$this->tableName}";
        } else {
            // استفاده از روش قدیمی
            $query = "SELECT 
                        COUNT(*) AS total_activities,
                        COUNT(DISTINCT chat_id) AS total_users,
                        SUM(CASE WHEN action_type = 'button_click' THEN 1 ELSE 0 END) AS total_button_clicks,
                        SUM(CASE WHEN action_type = 'section_view' THEN 1 ELSE 0 END) AS total_section_views,
                        SUM(CASE WHEN action_type = 'file_view' THEN 1 ELSE 0 END) AS total_file_views,
                        SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) AS activities_last_7_days,
                        COUNT(DISTINCT CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN chat_id END) AS active_users_last_7_days,
                        SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) AS activities_today,
                        COUNT(DISTINCT CASE WHEN DATE(created_at) = CURDATE() THEN chat_id END) AS active_users_today
                      FROM {$this->tableName}";
        }
        
        $result = $this->db->selectOne($query);
        
        return $result ?: [
            'total_activities' => 0,
            'total_users' => 0,
            'total_button_clicks' => 0,
            'total_section_views' => 0,
            'total_file_views' => 0,
            'activities_last_7_days' => 0,
            'active_users_last_7_days' => 0,
            'activities_today' => 0,
            'active_users_today' => 0,
        ];
    }

    /**
     * دریافت تمام شناسه‌های چت منحصربه‌فرد کاربران
     */
    public function getAllUniqueChatIds() {
        $query = "SELECT DISTINCT chat_id, username, first_name, last_name, MAX(created_at) as last_activity
                  FROM {$this->tableName}
                  GROUP BY chat_id, username, first_name, last_name
                  ORDER BY last_activity DESC";
        return $this->db->select($query);
    }
}

