<?php
require_once __DIR__ . '/../../config/database.php';

/**
 * کلاس مدیریت دیتابیس
 */
class Database {
    private $pdo;
    
    public function __construct() {
        // تنظیم timezone به ایران
        date_default_timezone_set('Asia/Tehran');
        $this->pdo = DatabaseConfig::getConnection();
        // تنظیم timezone برای MySQL
        $this->pdo->exec("SET time_zone = '+03:30'");
    }
    
    /**
     * دریافت اتصال PDO
     */
    public function getConnection() {
        return $this->pdo;
    }
    
    /**
     * اجرای کوئری SELECT
     */
    public function select($query, $params = []) {
        try {
            $stmt = $this->pdo->prepare($query);
            $stmt->execute($params);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Database select error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * اجرای کوئری SELECT برای یک رکورد
     */
    public function selectOne($query, $params = []) {
        try {
            if (empty($query)) {
                error_log("Database selectOne error: Empty query");
                return false;
            }
            $stmt = $this->pdo->prepare($query);
            $stmt->execute($params);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Database selectOne error: " . $e->getMessage() . " | Query: " . $query);
            return false;
        }
    }
    
    /**
     * اجرای کوئری INSERT
     */
    public function insert($query, $params = []) {
        try {
            $stmt = $this->pdo->prepare($query);
            $result = $stmt->execute($params);
            return $result ? $this->pdo->lastInsertId() : false;
        } catch (PDOException $e) {
            error_log("Database insert error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * اجرای کوئری UPDATE
     */
    public function update($query, $params = []) {
        try {
            if (empty($query)) {
                error_log("Database update error: Empty query");
                return false;
            }
            $stmt = $this->pdo->prepare($query);
            $result = $stmt->execute($params);
            if (!$result) {
                error_log("Database update error: Execution failed | Query: " . $query);
            }
            return $result;
        } catch (PDOException $e) {
            error_log("Database update error: " . $e->getMessage() . " | Query: " . $query);
            return false;
        }
    }
    
    /**
     * اجرای کوئری DELETE
     */
    public function delete($query, $params = []) {
        try {
            $stmt = $this->pdo->prepare($query);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            error_log("Database delete error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * شروع تراکنش
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * تایید تراکنش
     */
    public function commit() {
        return $this->pdo->commit();
    }
    
    /**
     * لغو تراکنش
     */
    public function rollback() {
        return $this->pdo->rollback();
    }
    
    /**
     * اجرای کوئری DDL (مثل ALTER TABLE)
     */
    public function execute($query) {
        try {
            return $this->pdo->exec($query);
        } catch (PDOException $e) {
            error_log("Database execute error: " . $e->getMessage() . " | Query: " . $query);
            return false;
        }
    }
}
