<?php
require_once __DIR__ . '/../TelegramBot/TelegramBot.php';
require_once __DIR__ . '/KeyboardManager.php';
require_once __DIR__ . '/../Models/AllModels.php';
require_once __DIR__ . '/../../config/bot.php';

/**
 * کلاس مدیریت جستجو
 */
class SearchHandler {
    private $bot;
    private $models;
    private $searchSessions = [];
    private $searchTexts = [];
    
    public function __construct() {
        $this->bot = new TelegramBot();
        $this->models = [
            'instructions' => new InstructionModel(),
            'regulations' => new RegulationModel(),
            'trainings' => new TrainingModel(),
            'reports' => new ReportModel(),
            'news' => new NewsModel(),
            'faq' => new FaqModel(),
            'ai' => new AiModel(),
            'forms' => new FormModel()
        ];
    }
    
    /**
     * مدیریت جستجو (برای webhook)
     */
    public function handleSearch($chatId, $messageId, $data) {
        $parts = explode('_', $data);
        
        if ($data === 'search_main') {
            // جستجوی عمومی
            $this->startGlobalSearch($chatId, $messageId);
        } elseif (count($parts) >= 2) {
            $section = $parts[1];
            $this->startSearch($chatId, $messageId, $section);
        } else {
            // جستجوی عمومی
            $this->startGlobalSearch($chatId, $messageId);
        }
    }
    
    /**
     * شروع جستجو در بخش خاص
     */
    public function startSearch($chatId, $messageId, $section) {
        if (!isset($this->models[$section])) {
            $this->bot->sendMessage($chatId, "🔍 جستجو برای این بخش در دسترس نیست.");
            return;
        }

        $text = BotConfig::getMessage('search_in_section') . " <b>" . $this->getSectionName($section) . "</b>\n\n";
        $text .= "لطفاً متن جستجو را وارد کنید:";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                ]
            ]
        ];
        
        $this->bot->sendMessage($chatId, $text, $keyboard);
        
        // ذخیره وضعیت جستجو
        $this->searchSessions[$chatId] = [
            'section' => $section,
            'message_id' => null // برای sendMessage نیازی به messageId نیست
        ];
    }
    
    /**
     * پردازش متن جستجو
     */
    public function processSearch($chatId, $searchText) {
        if (!isset($this->searchSessions[$chatId])) {
            return;
        }
        
        $session = $this->searchSessions[$chatId];
        $section = $session['section'];
        $messageId = $session['message_id'];
        
        // ذخیره متن جستجو
        $this->searchTexts[$chatId] = $searchText;
        
        if ($section === 'global') {
            // جستجوی عمومی
            $this->globalSearch($chatId, $searchText, $messageId, 1);
        } else {
            if (!isset($this->models[$section])) {
                $this->bot->sendMessage($chatId, "🔍 جستجو برای این بخش در دسترس نیست.");
                unset($this->searchSessions[$chatId], $this->searchTexts[$chatId]);
                return;
            }

            // جستجو در بخش مربوطه - 10 نتیجه در هر صفحه
            $searchLimit = BotConfig::SEARCH_RESULTS_PER_PAGE;
            $results = $this->models[$section]->search($searchText, $searchLimit, 0);
            $totalResults = $this->models[$section]->countSearch($searchText);
            $totalPages = ceil($totalResults / $searchLimit);
            
            $text = "🔍 <b>نتایج جستجو در " . $this->getSectionName($section) . "</b>\n";
            $text .= "📝 <i>جستجو برای: \"{$searchText}\"</i>\n\n";
            
            if (empty($results)) {
                $text .= BotConfig::getMessage('no_results');
            } else {
                $text .= "📄 <b>صفحه 1 از {$totalPages}</b>\n";
                $text .= "📊 <b>کل نتایج: {$totalResults}</b>";
            }
            
            $keyboard = KeyboardManager::getContentKeyboard($results, $section, 1, $totalPages, true);
            
            $this->bot->sendMessage($chatId, $text, $keyboard);
        }
        
        // session را نگه می‌داریم برای pagination (برای همه نوع جستجو)
        // searchText و session را نگه می‌داریم تا pagination کار کند
    }
    
    /**
     * شروع جستجوی عمومی
     */
    public function startGlobalSearch($chatId, $messageId) {
        $text = "🔍 <b>جستجو در همه فایل‌ها</b>\n\n";
        $text .= "لطفاً متن جستجو را وارد کنید:";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                ]
            ]
        ];
        
        $this->bot->sendMessage($chatId, $text, $keyboard);
        
        // ذخیره وضعیت جستجو عمومی
        $this->searchSessions[$chatId] = [
            'section' => 'global',
            'message_id' => null // برای sendMessage نیازی به messageId نیست
        ];
    }
    
    /**
     * مدیریت پیجینیشن جستجو
     */
    public function handleSearchPagination($chatId, $messageId, $data, $messageText = null) {
        $parts = explode('_', $data);
        $section = $parts[1];
        $page = (int)$parts[3];
        
        // دریافت متن جستجو
        $searchText = $this->getSearchText($chatId, $section);
        
        // اگر متن جستجو پیدا نشد، از پیام قبلی استخراج می‌کنیم
        if ((!$searchText || $searchText === "جستجو") && $messageText) {
            // استخراج متن جستجو از پیام: 📝 <i>جستجو برای: "متن جستجو"</i>
            // پاک کردن HTML tags برای جستجوی بهتر
            $cleanText = strip_tags($messageText);
            if (preg_match('/جستجو برای:\s*"([^"]+)"/u', $cleanText, $matches)) {
                $searchText = $matches[1];
                // ذخیره برای استفاده بعدی
                $this->searchTexts[$chatId] = $searchText;
            } elseif (preg_match('/جستجو برای:\s*([^\n]+)/u', $cleanText, $matches)) {
                // اگر فرمت دیگری داشت
                $searchText = trim($matches[1]);
                // حذف کاراکترهای اضافی
                $searchText = trim($searchText, '"\'');
                $this->searchTexts[$chatId] = $searchText;
            }
        }
        
        if (!$searchText || $searchText === "جستجو") {
            $this->bot->sendMessage($chatId, "خطا در دریافت اطلاعات جستجو");
            return;
        }
        
        // اگر global search است
        if ($section === 'global') {
            $this->globalSearch($chatId, $searchText, $messageId, $page);
            return;
        }
        
        if (!isset($this->models[$section])) {
            $this->bot->sendMessage($chatId, "🔍 جستجو برای این بخش در دسترس نیست.");
            return;
        }
        
        // جستجو در صفحه جدید - 10 نتیجه در هر صفحه
        $searchLimit = BotConfig::SEARCH_RESULTS_PER_PAGE;
        $offset = ($page - 1) * $searchLimit;
        $results = $this->models[$section]->search($searchText, $searchLimit, $offset);
        $totalResults = $this->models[$section]->countSearch($searchText);
        $totalPages = ceil($totalResults / $searchLimit);
        
        $text = "🔍 <b>نتایج جستجو در " . $this->getSectionName($section) . "</b>\n";
        $text .= "📝 <i>جستجو برای: \"{$searchText}\"</i>\n\n";
        
        if (empty($results)) {
            $text .= BotConfig::getMessage('no_results');
        } else {
            $text .= "📄 <b>صفحه {$page} از {$totalPages}</b>\n";
            $text .= "📊 <b>کل نتایج: {$totalResults}</b>";
        }
        
        // ذخیره searchText برای pagination
        $this->searchTexts[$chatId] = $searchText;
        
        $keyboard = KeyboardManager::getContentKeyboard($results, $section, $page, $totalPages, true);
        
        $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
    }
    
    /**
     * جستجوی عمومی در همه بخش‌ها
     */
    public function globalSearch($chatId, $searchText, $messageId = null, $page = 1) {
        $allResults = [];
        
        // دریافت همه نتایج از همه بخش‌ها (بدون محدودیت)
        foreach ($this->models as $section => $model) {
            if ($section === 'ethics') {
                continue;
            }
            $results = $model->search($searchText); // همه نتایج بدون محدودیت
            foreach ($results as $result) {
                $result['section'] = $section;
                $result['section_name'] = $this->getSectionName($section);
                $allResults[] = $result;
            }
        }
        
        // ذخیره searchText برای pagination
        $this->searchTexts[$chatId] = $searchText;
        
        // مرتب‌سازی بر اساس relevance (پین شده‌ها اول)
        usort($allResults, function($a, $b) {
            $pinnedA = !empty($a['is_pinned']) ? 1 : 0;
            $pinnedB = !empty($b['is_pinned']) ? 1 : 0;
            if ($pinnedA !== $pinnedB) {
                return $pinnedB - $pinnedA; // پین شده‌ها اول
            }
            return 0;
        });
        
        $totalResults = count($allResults);
        $searchLimit = BotConfig::SEARCH_RESULTS_PER_PAGE;
        $totalPages = ceil($totalResults / $searchLimit);
        
        if (empty($allResults)) {
            $text = "🔍 <b>نتایج جستجو</b>\n\n";
            $text .= "📝 <i>جستجو برای: \"{$searchText}\"</i>\n\n";
            $text .= BotConfig::getMessage('no_results');
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                    ]
                ]
            ];
            
            if ($messageId) {
                $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
            } else {
                $this->bot->sendMessage($chatId, $text, $keyboard);
            }
            
            return;
        }
        
        // صفحه‌بندی
        $offset = ($page - 1) * $searchLimit;
        $pageResults = array_slice($allResults, $offset, $searchLimit);
        
        $text = "🔍 <b>نتایج جستجو</b>\n\n";
        $text .= "📝 <i>جستجو برای: \"{$searchText}\"</i>\n\n";
        
        if ($totalPages > 1) {
            $text .= "📄 <b>صفحه {$page} از {$totalPages}</b>\n";
        }
        $text .= "📊 <b>کل نتایج: {$totalResults}</b>";
        
        $keyboard = [
            'inline_keyboard' => []
        ];
        
        foreach ($pageResults as $result) {
            $displayTitle = $result['section'] === 'faq' ? $result['question'] : ($result['title'] ?? 'بدون عنوان');
            // تقسیم عنوان طولانی به دو خط
            $displayTitle = KeyboardManager::wrapLongText($displayTitle, 40);
            // اضافه کردن اموجی پین اگر پین شده باشد
            $pinnedIcon = !empty($result['is_pinned']) ? '📌 ' : '';
            $keyboard['inline_keyboard'][] = [
                ['text' => $pinnedIcon . $displayTitle . " ({$result['section_name']})", 'callback_data' => "content_{$result['section']}_{$result['id']}"]
            ];
        }
        
        // دکمه‌های pagination
        if ($totalPages > 1) {
            $paginationButtons = [];
            
            if ($page > 1) {
                $paginationButtons[] = ['text' => '<', 'callback_data' => "search_global_page_" . ($page - 1)];
            }
            
            if ($page < $totalPages) {
                $paginationButtons[] = ['text' => '>', 'callback_data' => "search_global_page_" . ($page + 1)];
            }
            
            if (!empty($paginationButtons)) {
                $keyboard['inline_keyboard'][] = $paginationButtons;
            }
        }
        
        $keyboard['inline_keyboard'][] = [
            ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
        ];
        
        if ($messageId) {
            $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->bot->sendMessage($chatId, $text, $keyboard);
        }
    }
    
    /**
     * دریافت نام بخش
     */
    private function getSectionName($section) {
        $names = [
            'instructions' => BotConfig::getMessage('instructions'),
            'regulations' => BotConfig::getMessage('regulations'),
            'trainings' => BotConfig::getMessage('trainings'),
            'reports' => BotConfig::getMessage('reports'),
            'news' => BotConfig::getMessage('news'),
            'faq' => BotConfig::getMessage('faq'),
            'ai' => BotConfig::getMessage('ai'),
            'ethics' => BotConfig::getMessage('ethics'),
            'forms' => BotConfig::getMessage('forms')
        ];
        
        return $names[$section] ?? $section;
    }
    
    /**
     * دریافت متن جستجو
     */
    private function getSearchText($chatId, $section = null) {
        return $this->searchTexts[$chatId] ?? "جستجو";
    }
    
    /**
     * بررسی وجود session جستجو
     */
    public function hasSearchSession($chatId) {
        return isset($this->searchSessions[$chatId]);
    }
    
    /**
     * حذف session جستجو
     */
    public function clearSearchSession($chatId) {
        unset($this->searchSessions[$chatId]);
        unset($this->searchTexts[$chatId]);
    }
}
