<?php
require_once __DIR__ . '/../../config/bot.php';

/**
 * کلاس مدیریت کیبوردهای بات
 */
class KeyboardManager {
    
    /**
     * ایجاد کیبورد منوی اصلی
     */
    public static function getMainMenuKeyboard() {
        return [
            'inline_keyboard' => [
                [
                    ['text' => BotConfig::getMessage('search'), 'callback_data' => 'search_main']
                ],
                [
                    ['text' => BotConfig::getMessage('regulations'), 'callback_data' => 'menu_regulations'],
                    ['text' => BotConfig::getMessage('instructions'), 'callback_data' => 'menu_instructions']
                ],
                [
                    ['text' => BotConfig::getMessage('reports'), 'callback_data' => 'menu_reports'],
                    ['text' => BotConfig::getMessage('trainings'), 'callback_data' => 'menu_trainings']
                ],
                [
                    ['text' => BotConfig::getMessage('news'), 'callback_data' => 'menu_news'],
                    ['text' => BotConfig::getMessage('forms'), 'callback_data' => 'menu_forms']
                ],
                [
                    ['text' => BotConfig::getMessage('ai'), 'callback_data' => 'menu_ai'],
                    ['text' => BotConfig::getMessage('ethics'), 'callback_data' => 'menu_ethics']
                ],
                [
                    ['text' => BotConfig::getMessage('faq'), 'callback_data' => 'menu_faq']
                ],
                [
                    ['text' => BotConfig::getMessage('about_us'), 'callback_data' => 'about_us']
                ]
            ]
        ];
    }
    
    /**
     * ایجاد کیبورد ثابت (Reply Keyboard) برای منوی اصلی
     */
    public static function getReplyKeyboard() {
        return [
            'keyboard' => [
                [
                    ['text' => '🏠 منوی اصلی']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
    }
    
    /**
     * ایجاد کیبورد بخش‌های مختلف
     */
    public static function getSectionKeyboard($section, $page = 1, $isSearch = false) {
        $keyboard = [
            'inline_keyboard' => []
        ];
        
        // دکمه جستجو
        $searchCallback = $isSearch ? "search_{$section}" : "search_{$section}";
        $keyboard['inline_keyboard'][] = [
            ['text' => BotConfig::getMessage('search'), 'callback_data' => $searchCallback]
        ];
        
        return $keyboard;
    }
    
    /**
     * ایجاد کیبورد لیست مطالب با پیجینیشن
     */
    public static function getContentKeyboard($items, $section, $page = 1, $totalPages = 1, $isSearch = false) {
        $keyboard = [
            'inline_keyboard' => []
        ];
        
        // دکمه جستجو
        $searchCallback = $isSearch ? "search_{$section}" : "search_{$section}";
        $keyboard['inline_keyboard'][] = [
            ['text' => BotConfig::getMessage('search'), 'callback_data' => $searchCallback]
        ];
        
        // دکمه‌های مطالب
        foreach ($items as $item) {
            $text = $section === 'faq' ? $item['question'] : $item['title'];
            // تقسیم عنوان طولانی به دو خط (حداکثر 40 کاراکتر در هر خط)
            $text = self::wrapLongText($text, 40);
            // اضافه کردن اموجی پین اگر پین شده باشد
            $pinnedIcon = (!empty($item['is_pinned'])) ? '📌 ' : '';
            $keyboard['inline_keyboard'][] = [
                ['text' => $pinnedIcon . $text, 'callback_data' => "content_{$section}_{$item['id']}"]
            ];
        }
        
        // دکمه‌های پیجینیشن
        if ($totalPages > 1) {
            $paginationButtons = [];
            
            if ($page > 1) {
                $prevCallback = $isSearch ? "search_{$section}_page_" . ($page - 1) : "{$section}_page_" . ($page - 1);
                $paginationButtons[] = ['text' => '<', 'callback_data' => $prevCallback];
            }
            
            if ($page < $totalPages) {
                $nextCallback = $isSearch ? "search_{$section}_page_" . ($page + 1) : "{$section}_page_" . ($page + 1);
                $paginationButtons[] = ['text' => '>', 'callback_data' => $nextCallback];
            }
            
            if (!empty($paginationButtons)) {
                $keyboard['inline_keyboard'][] = $paginationButtons;
            }
        }
        
        // دکمه بازگشت به منوی اصلی
        $keyboard['inline_keyboard'][] = [
            ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
        ];
        
        return $keyboard;
    }
    
    /**
     * ایجاد کیبورد با دسته‌بندی و فایل‌ها
     */
    public static function getContentKeyboardWithCategories($categories, $files, $section, $parentId = null, $parentFolderId = null) {
        $keyboard = [
            'inline_keyboard' => []
        ];
        
        // دکمه جستجو
        $keyboard['inline_keyboard'][] = [
            ['text' => BotConfig::getMessage('search'), 'callback_data' => "search_{$section}"]
        ];
        
        // نمایش فولدرها به صورت جدا
        if (!empty($categories)) {
            // هدر فولدرها (فقط برای نمایش بهتر - نمی‌توانیم متن ساده اضافه کنیم پس از آیکون استفاده می‌کنیم)
            foreach ($categories as $category) {
                $text = "📁 " . htmlspecialchars($category['title']);
                $keyboard['inline_keyboard'][] = [
                    ['text' => $text, 'callback_data' => "category_{$section}_{$category['id']}"]
                ];
            }
        }
        
        // نمایش فایل‌ها به صورت جدا
        if (!empty($files)) {
            foreach ($files as $file) {
                $text = $section === 'faq' ? $file['question'] : htmlspecialchars($file['title']);
                // تقسیم عنوان طولانی به دو خط (حداکثر 40 کاراکتر در هر خط)
                $text = self::wrapLongText($text, 40);
                // اضافه کردن اموجی پین اگر پین شده باشد
                $pinnedIcon = (!empty($file['is_pinned'])) ? '📌 ' : '';
                $keyboard['inline_keyboard'][] = [
                    ['text' => $pinnedIcon . $text, 'callback_data' => "content_{$section}_{$file['id']}"]
                ];
            }
        }
        
        // دکمه بازگشت
        if ($parentId !== null) {
            // اگر parentFolderId داریم، به فولدر والد برمی‌گردیم، وگرنه به سطح اصلی
            if ($parentFolderId !== null) {
                $keyboard['inline_keyboard'][] = [
                    ['text' => '◀️ بازگشت', 'callback_data' => "category_{$section}_{$parentFolderId}"]
                ];
            } else {
                $keyboard['inline_keyboard'][] = [
                    ['text' => '◀️ بازگشت', 'callback_data' => "menu_{$section}"]
                ];
            }
        } else {
            // بازگشت به منوی اصلی
            $keyboard['inline_keyboard'][] = [
                ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
            ];
        }
        
        return $keyboard;
    }
    
    /**
     * ایجاد کیبورد جستجو
     */
    public static function getSearchKeyboard($section) {
        return [
            'inline_keyboard' => [
                [
                    ['text' => '🔍 جستجو', 'callback_data' => "search_input_{$section}"]
                ],
                [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                ]
            ]
        ];
    }
    
    /**
     * ایجاد کیبورد تایید
     */
    public static function getConfirmationKeyboard($action, $id) {
        return [
            'inline_keyboard' => [
                [
                    ['text' => '✅ تایید', 'callback_data' => "confirm_{$action}_{$id}"],
                    ['text' => '❌ لغو', 'callback_data' => 'cancel']
                ]
            ]
        ];
    }
    
    /**
     * ایجاد کیبورد خالی (حذف کیبورد)
     */
    public static function getEmptyKeyboard() {
        return [
            'inline_keyboard' => []
        ];
    }
    
    /**
     * تقسیم متن طولانی به دو خط
     * تلگرام به صورت خودکار متن را wrap می‌کند، اما برای اطمینان از نمایش درست،
     * متن را به دو خط تقسیم می‌کنیم
     */
    public static function wrapLongText($text, $maxLength = 35) {
        // حذف HTML tags اگر وجود داشته باشد
        $text = strip_tags($text);
        
        if (mb_strlen($text) <= $maxLength) {
            return $text;
        }
        
        // پیدا کردن بهترین نقطه برای تقسیم (اولین فاصله بعد از نیمه)
        $half = (int)($maxLength * 0.65); // 65% از طول برای خط اول
        $bestBreak = $half;
        
        // جستجوی فاصله یا کاراکتر مناسب برای تقسیم
        for ($i = $half; $i < min(mb_strlen($text), $maxLength + 15); $i++) {
            $char = mb_substr($text, $i, 1);
            if ($char === ' ' || $char === '،' || $char === ',' || $char === '.' || $char === '؛' || $char === '-' || $char === '_') {
                $bestBreak = $i + 1;
                break;
            }
        }
        
        // اگر نقطه مناسب پیدا نشد، از نیمه تقسیم می‌کنیم
        if ($bestBreak === $half) {
            $bestBreak = $maxLength;
        }
        
        $line1 = mb_substr($text, 0, $bestBreak);
        $line2 = mb_substr($text, $bestBreak);
        
        // اگر خط دوم خیلی کوتاه است، آن را به خط اول اضافه می‌کنیم
        if (mb_strlen($line2) < 5 && mb_strlen($text) <= $maxLength * 1.5) {
            return $text;
        }
        
        // اطمینان از اینکه خط دوم خیلی طولانی نباشد
        if (mb_strlen($line2) > $maxLength) {
            $line2 = mb_substr($line2, 0, $maxLength - 3) . '...';
        }
        
        return trim($line1) . "\n" . trim($line2);
    }
}
