<?php
require_once __DIR__ . '/../TelegramBot/TelegramBot.php';
require_once __DIR__ . '/KeyboardManager.php';
require_once __DIR__ . '/SearchHandler.php';
require_once __DIR__ . '/../Models/AllModels.php';
require_once __DIR__ . '/../../config/bot.php';

/**
 * کلاس اصلی مدیریت بات
 */
class BotHandler {
    private $bot;
    private $models;
    private $searchHandler;
    private $settingsModel;
    private $activityLogModel;
    private $botUserModel;
    private $channelChatId;
    
    public function __construct() {
        $this->bot = new TelegramBot();
        $this->searchHandler = new SearchHandler();
        $this->settingsModel = new SettingsModel();
        $this->activityLogModel = new ActivityLogModel();
        $this->botUserModel = new BotUserModel();
        $this->models = [
            'instructions' => new InstructionModel(),
            'regulations' => new RegulationModel(),
            'trainings' => new TrainingModel(),
            'reports' => new ReportModel(),
            'news' => new NewsModel(),
            'faq' => new FaqModel(),
            'ai' => new AiModel(),
            'ethics' => new EthicModel(),
            'forms' => new FormModel()
        ];
        $this->channelChatId = $this->resolveChannelChatId(BotConfig::TELEGRAM_CHANNEL);
    }
    
    /**
     * ثبت یا به‌روزرسانی کاربر و ثبت لاگ فعالیت
     */
    private function logActivity($chatId, $actionType, $actionName, $section = null, $itemId = null, $itemTitle = null, $metadata = [], $userInfo = []) {
        try {
            // ثبت یا به‌روزرسانی کاربر در جدول bot_users
            $userId = null;
            
            if (!empty($userInfo) || !empty($chatId)) {
                try {
                    // بررسی وجود کاربر قبل از ثبت
                    $existingUser = $this->botUserModel->getByChatId($chatId);
                    $isNewUser = empty($existingUser);
                    
                    if ($isNewUser) {
                        // ثبت کاربر جدید (activity_count = 1 در registerOrUpdateUser تنظیم می‌شود)
                        $userId = $this->botUserModel->registerOrUpdateUser($chatId, $userInfo);
                        
                        // اگر ID دریافت نکردیم، دوباره پیدا می‌کنیم
                        if (!$userId || !is_numeric($userId)) {
                            $user = $this->botUserModel->getByChatId($chatId);
                            $userId = $user['id'] ?? null;
                        }
                    } else {
                        // کاربر قدیمی: به‌روزرسانی اطلاعات و افزایش تعداد فعالیت
                        // اول اطلاعات را به‌روزرسانی می‌کنیم
                        $updateData = [
                            'username' => !empty($userInfo['username']) ? $userInfo['username'] : $existingUser['username'],
                            'first_name' => !empty($userInfo['first_name']) ? $userInfo['first_name'] : $existingUser['first_name'],
                            'last_name' => !empty($userInfo['last_name']) ? $userInfo['last_name'] : $existingUser['last_name'],
                            'last_seen_at' => date('Y-m-d H:i:s')
                        ];
                        $this->botUserModel->updateByChatId($chatId, $updateData);
                        
                        // سپس تعداد فعالیت را افزایش می‌دهیم
                        $this->botUserModel->incrementActivity($chatId);
                        
                        $userId = $existingUser['id'] ?? null;
                    }
                } catch (Exception $e) {
                    error_log("Error registering/updating user: " . $e->getMessage());
                }
            }
            
            // ثبت لاگ فعالیت
            $logData = [
                'chat_id' => $chatId,
                'user_id' => $userId,
                'username' => $userInfo['username'] ?? null,
                'first_name' => $userInfo['first_name'] ?? null,
                'last_name' => $userInfo['last_name'] ?? null,
                'action_type' => $actionType,
                'action_name' => $actionName,
                'section' => $section,
                'item_id' => $itemId,
                'item_title' => $itemTitle,
                'metadata' => $metadata
            ];
            
            $this->activityLogModel->logActivity($logData);
        } catch (Exception $e) {
            // در صورت خطا، لاگ را ثبت می‌کنیم اما برنامه ادامه می‌یابد
            error_log("Error logging activity: " . $e->getMessage());
        }
    }
    
    /**
     * پردازش پیام دریافتی
     */
    public function handleUpdate($update) {
        if (isset($update['message'])) {
            $this->handleMessage($update['message']);
        } elseif (isset($update['callback_query'])) {
            $this->handleCallbackQuery($update['callback_query']);
        }
    }
    
    /**
     * پردازش پیام متنی
     */
    private function handleMessage($message) {
        $chatId = $message['chat']['id'];
        $text = $message['text'] ?? '';
        $userInfo = [
            'username' => $message['chat']['username'] ?? null,
            'first_name' => $message['chat']['first_name'] ?? null,
            'last_name' => $message['chat']['last_name'] ?? null
        ];
        $userId = $message['from']['id'] ?? null;

        if (!$this->ensureChannelMembership($chatId, $userId)) {
            return;
        }
        
        // بررسی دستورات خاص
        if ($text === '/start') {
            $this->searchHandler->clearSearchSession($chatId);
            // ثبت لاگ
            $this->logActivity($chatId, 'button_click', 'start', null, null, null, [], $userInfo);
            // ارسال منوی اصلی (با Reply Keyboard)
            $this->sendMainMenu($chatId, null, true);
        } elseif ($text === '/help') {
            $this->searchHandler->clearSearchSession($chatId);
            $this->sendHelp($chatId);
        } elseif ($text === '🏠 منوی اصلی' || $text === 'منوی اصلی') {
            // اگر کاربر دکمه منوی اصلی را فشار داد
            $this->searchHandler->clearSearchSession($chatId);
            // ثبت لاگ
            $this->logActivity($chatId, 'button_click', 'main_menu', null, null, null, [], $userInfo);
            $this->sendMainMenu($chatId);
        } else {
            // بررسی وجود session جستجو
            if ($this->searchHandler->hasSearchSession($chatId)) {
                $this->searchHandler->processSearch($chatId, $text);
            }
            // اگر session جستجو وجود ندارد و پیام دستور خاصی نیست، کاری نمی‌کنیم
        }
    }
    
    /**
     * پردازش callback query
     */
    private function handleCallbackQuery($callbackQuery) {
        $chatId = $callbackQuery['message']['chat']['id'];
        $messageId = $callbackQuery['message']['message_id'];
        $data = $callbackQuery['data'];
        $userInfo = [
            'username' => $callbackQuery['from']['username'] ?? null,
            'first_name' => $callbackQuery['from']['first_name'] ?? null,
            'last_name' => $callbackQuery['from']['last_name'] ?? null
        ];
        $userId = $callbackQuery['from']['id'] ?? null;

        if ($data === 'check_membership') {
            $isMember = $this->checkChannelMembership($userId);
            if ($isMember) {
                $this->bot->sendRequest('answerCallbackQuery', [
                    'callback_query_id' => $callbackQuery['id'],
                    'text' => 'عضویت شما تایید شد ✅',
                    'show_alert' => false
                ]);

                $successText = BotConfig::getMessage('channel_membership_confirmed');
                $keyboard = [
                    'inline_keyboard' => [
                        [
                            ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                        ]
                    ]
                ];
                $this->bot->editMessageText($chatId, $messageId, $successText, $keyboard, true);
                $this->sendMainMenu($chatId, null, true);
            } else {
                $this->bot->sendRequest('answerCallbackQuery', [
                    'callback_query_id' => $callbackQuery['id'],
                    'text' => BotConfig::getMessage('channel_membership_pending'),
                    'show_alert' => true
                ]);
                $this->sendChannelJoinPrompt($chatId);
            }
            return;
        }

        if (!$this->ensureChannelMembership($chatId, $userId)) {
            $this->bot->sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callbackQuery['id'],
                'text' => 'لطفاً ابتدا عضو کانال شوید.',
                'show_alert' => true
            ]);
            return;
        }
        
        // پاسخ به callback query
        $this->bot->sendRequest('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id']
        ]);
        
        // پردازش بر اساس نوع callback
        if ($data === 'main_menu') {
            $this->searchHandler->clearSearchSession($chatId);
            $this->logActivity($chatId, 'button_click', 'main_menu', null, null, null, [], $userInfo);
            $this->sendMainMenu($chatId, $messageId);
        } elseif (strpos($data, 'menu_') === 0) {
            $section = substr($data, 5);
            $this->logActivity($chatId, 'button_click', 'menu_' . $section, $section, null, null, [], $userInfo);
            if ($section === 'ethics') {
                $this->showEthicsQuote($chatId, $messageId, $userInfo);
            } else {
                $this->showSection($chatId, $messageId, $section, null, $userInfo);
            }
        } elseif ($data === 'ethics_next') {
            $this->showEthicsQuote($chatId, $messageId, $userInfo, true);
        } elseif (strpos($data, 'category_') === 0) {
            // نمایش محتویات یک دسته
            // فرمت: category_section_id
            $parts = explode('_', $data);
            if (count($parts) >= 3) {
                $section = $parts[1];
                $parentId = (int)$parts[2]; // تبدیل به عدد
                $this->logActivity($chatId, 'section_view', 'category', $section, $parentId, null, [], $userInfo);
                $this->showSection($chatId, $messageId, $section, $parentId, $userInfo);
            }
        } elseif (strpos($data, 'content_') === 0) {
            $this->handleContent($chatId, $messageId, $data, $userInfo);
        } elseif (strpos($data, '_page_') !== false) {
            $this->handlePagination($chatId, $messageId, $data);
        } elseif (strpos($data, 'search_') === 0) {
            $this->searchHandler->handleSearch($chatId, $messageId, $data);
        } elseif (strpos($data, 'search_page_') === 0) {
            $messageText = $callbackQuery['message']['text'] ?? '';
            $this->searchHandler->handleSearchPagination($chatId, $messageId, $data, $messageText);
        } elseif ($data === 'about_us') {
            $this->logActivity($chatId, 'button_click', 'about_us', null, null, null, [], $userInfo);
            $this->showAboutUs($chatId, $messageId);
        }
    }
    
    /**
     * ارسال منوی اصلی
     */
    private function sendMainMenu($chatId, $messageId = null, $withReplyKeyboard = false) {
        $text = "<b>" . BotConfig::getMessage('main_menu') . "</b>\n\n" . BotConfig::getMessage('welcome');
        $keyboard = KeyboardManager::getMainMenuKeyboard();
        $replyKeyboard = null;
        
        // اگر نیاز به Reply Keyboard باشد، آن را تنظیم می‌کنیم
        if ($withReplyKeyboard) {
            $replyKeyboard = KeyboardManager::getReplyKeyboard();
        }
        
        if ($messageId) {
            $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->bot->sendMessage($chatId, $text, $keyboard, $replyKeyboard);
        }
    }
    
    /**
     * نمایش بخش خاص
     */
    private function showSection($chatId, $messageId, $section, $parentId = null, $userInfo = []) {
        $sectionNames = [
            'instructions' => BotConfig::getMessage('instructions'),
            'regulations' => BotConfig::getMessage('regulations'),
            'trainings' => BotConfig::getMessage('trainings'),
            'reports' => BotConfig::getMessage('reports'),
            'news' => BotConfig::getMessage('news'),
            'faq' => BotConfig::getMessage('faq'),
            'ai' => BotConfig::getMessage('ai'),
            'ethics' => BotConfig::getMessage('ethics'),
            'forms' => BotConfig::getMessage('forms')
        ];
        
        // ثبت لاگ مشاهده بخش (اگر userInfo موجود نباشد، لاگ ثبت نمی‌کنیم)
        if (!empty($userInfo)) {
            $this->logActivity($chatId, 'section_view', 'section_open', $section, $parentId, null, [], $userInfo);
        }
        
        // برای FAQ و AI دسته‌بندی نداریم
        if ($section === 'faq' || $section === 'ai') {
            $title = $sectionNames[$section] ?? $section;
            $text = "<b>{$title}</b>\n\n";
            
            // دریافت مطالب
            $items = $this->models[$section]->getPaginated(1, BotConfig::MAX_RESULTS_PER_PAGE);
            $totalPages = $this->models[$section]->getTotalPages(BotConfig::MAX_RESULTS_PER_PAGE);
            
            if (empty($items)) {
                $text .= BotConfig::getMessage('no_results');
            } else {
                $text .= "📄 <b>صفحه 1 از {$totalPages}</b>";
            }
            
            $keyboard = KeyboardManager::getContentKeyboard($items, $section, 1, $totalPages);
            $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }
        
        // دریافت تمام آیتم‌ها برای جداسازی دسته‌ها و فایل‌ها
        $allItems = $this->models[$section]->getAll();
        
        // دسته‌ها و فایل‌ها - بر اساس item_type و parent_id
        if ($parentId === null) {
            // در سطح اصلی: فقط فولدرها و فایل‌هایی که parent_id = NULL
            $categories = array_filter($allItems, function($item) {
                $itemType = $item['item_type'] ?? 'file';
                // بررسی دقیق: parent_id باید NULL باشد (نه 0، نه string خالی)
                $itemParentId = $item['parent_id'] ?? null;
                $isRoot = ($itemParentId === null || $itemParentId === '');
                return $itemType === 'folder' && $isRoot;
            });
            $files = array_filter($allItems, function($item) {
                $itemType = $item['item_type'] ?? 'file';
                // بررسی دقیق: parent_id باید NULL باشد (نه 0، نه string خالی)
                $itemParentId = $item['parent_id'] ?? null;
                $isRoot = ($itemParentId === null || $itemParentId === '');
                return $itemType === 'file' && $isRoot;
            });
        } else {
            // درون یک دسته: فقط فولدرها و فایل‌هایی که parent_id = $parentId
            $parentId = (int)$parentId; // تبدیل به عدد برای مقایسه دقیق
            $categories = array_filter($allItems, function($item) use ($parentId) {
                $itemType = $item['item_type'] ?? 'file';
                // تبدیل parent_id به عدد برای مقایسه دقیق
                $itemParentId = isset($item['parent_id']) && $item['parent_id'] !== null && $item['parent_id'] !== '' 
                    ? (int)$item['parent_id'] 
                    : null;
                return $itemType === 'folder' && $itemParentId !== null && $itemParentId === $parentId;
            });
            $files = array_filter($allItems, function($item) use ($parentId) {
                $itemType = $item['item_type'] ?? 'file';
                // تبدیل parent_id به عدد برای مقایسه دقیق
                $itemParentId = isset($item['parent_id']) && $item['parent_id'] !== null && $item['parent_id'] !== '' 
                    ? (int)$item['parent_id'] 
                    : null;
                return $itemType === 'file' && $itemParentId !== null && $itemParentId === $parentId;
            });
        }
        
        // مرتب‌سازی بر اساس is_pinned و display_order (پین شده‌ها اول)
        usort($categories, function($a, $b) {
            // اول پین شده‌ها
            $pinnedA = !empty($a['is_pinned']) ? 1 : 0;
            $pinnedB = !empty($b['is_pinned']) ? 1 : 0;
            if ($pinnedA !== $pinnedB) {
                return $pinnedB - $pinnedA; // پین شده‌ها اول
            }
            
            // سپس display_order
            $orderA = isset($a['display_order']) ? (int)$a['display_order'] : 0;
            $orderB = isset($b['display_order']) ? (int)$b['display_order'] : 0;
            if ($orderA === $orderB) {
                // اگر display_order یکسان است، بر اساس id مرتب می‌کنیم
                return (int)$a['id'] - (int)$b['id'];
            }
            return $orderA - $orderB;
        });
        usort($files, function($a, $b) {
            // اول پین شده‌ها
            $pinnedA = !empty($a['is_pinned']) ? 1 : 0;
            $pinnedB = !empty($b['is_pinned']) ? 1 : 0;
            if ($pinnedA !== $pinnedB) {
                return $pinnedB - $pinnedA; // پین شده‌ها اول
            }
            
            // سپس display_order
            $orderA = isset($a['display_order']) ? (int)$a['display_order'] : 0;
            $orderB = isset($b['display_order']) ? (int)$b['display_order'] : 0;
            if ($orderA === $orderB) {
                // اگر display_order یکسان است، بر اساس id مرتب می‌کنیم
                return (int)$a['id'] - (int)$b['id'];
            }
            return $orderA - $orderB;
        });
        
        // عنوان
        $title = $sectionNames[$section] ?? $section;
        $text = "<b>{$title}</b>\n";
        
        // اگر در یک دسته هستیم، عنوان آن را نشان می‌دهیم
        $parentFolderId = null;
        if ($parentId !== null) {
            $parentItem = $this->models[$section]->getById($parentId);
            if ($parentItem) {
                $text .= "📁 <b>" . htmlspecialchars($parentItem['title']) . "</b>\n";
                // parent_id فولدر فعلی برای دکمه بازگشت
                $parentFolderId = $parentItem['parent_id'] ?? null;
            }
        }
        
        $text .= "\n";
        
        // نمایش تعداد فولدرها و فایل‌ها
        $categoriesCount = count($categories);
        $filesCount = count($files);
        
        if ($categoriesCount > 0 || $filesCount > 0) {
            if ($categoriesCount > 0) {
                $text .= "📁 <b>فولدرها ({$categoriesCount}):</b>\n";
            }
            if ($filesCount > 0) {
                $text .= "📄 <b>فایل‌ها ({$filesCount}):</b>\n";
            }
        } else {
            $text .= BotConfig::getMessage('no_results');
        }
        
        $keyboard = KeyboardManager::getContentKeyboardWithCategories($categories, $files, $section, $parentId, $parentFolderId);
        
        $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
    }
    
    /**
     * نمایش جمله تصادفی اخلاق مهندسی
     */
    private function showEthicsQuote($chatId, $messageId = null, $userInfo = [], $isRefresh = false) {
        if (!isset($this->models['ethics'])) {
            return;
        }

        $quote = $this->models['ethics']->getRandomQuote();

        if (!$quote) {
            $text = "🧭 <b>" . BotConfig::getMessage('ethics') . "</b>\n\n";
            $text .= "❌ هنوز جمله‌ای برای نمایش ثبت نشده است.\n";
            $text .= "لطفاً از طریق پنل مدیریت جمله‌ای اضافه کنید.";

            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                    ]
                ]
            ];

            if ($messageId) {
                $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
            } else {
                $this->bot->sendMessage($chatId, $text, $keyboard);
            }
            return;
        }

        $quoteText = htmlspecialchars($quote['quote_text'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        $text = "🧭 <b>" . BotConfig::getMessage('ethics') . "</b>\n\n";
        $text .= "«{$quoteText}»";

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔁 جمله دیگر', 'callback_data' => 'ethics_next']
                ],
                [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                ]
            ]
        ];

        $metadata = [
            'quote_length' => mb_strlen($quote['quote_text'] ?? ''),
            'refreshed' => $isRefresh
        ];
        $snippet = mb_substr($quote['quote_text'], 0, 120);

        $this->logActivity(
            $chatId,
            'section_view',
            $isRefresh ? 'ethics_quote_refresh' : 'ethics_quote',
            'ethics',
            $quote['id'] ?? null,
            $snippet,
            $metadata,
            $userInfo
        );

        if ($messageId) {
            $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->bot->sendMessage($chatId, $text, $keyboard);
        }
    }

    /**
     * مدیریت محتوا
     */
    private function handleContent($chatId, $messageId, $data, $userInfo = []) {
        $parts = explode('_', $data);
        $section = $parts[1];
        $contentId = $parts[2];
        
        $item = $this->models[$section]->getById($contentId);
        
        if (!$item) {
            $this->bot->sendMessage($chatId, BotConfig::getMessage('file_not_found'));
            return;
        }
        
        // ثبت لاگ مشاهده محتوا
        $itemTitle = $this->getItemTitle($item, $section);
        $this->logActivity($chatId, 'file_view', 'content_view', $section, $contentId, $itemTitle, [], $userInfo);
        
        if ($section === 'faq') {
            // برای FAQ، پاسخ را نمایش می‌دهیم
            $text = "❓ <b>{$item['question']}</b>\n\n";
            $text .= "💡 <b>پاسخ:</b>\n";
            $text .= $item['answer'];
            
            $this->bot->sendMessage($chatId, $text);
        } elseif ($section === 'ai') {
            // برای AI، لینک وبسایت را ارسال می‌کنیم
            if ($item['website_url']) {
                $text = "🤖 <b>{$item['title']}</b>\n\n";
                $text .= "🌐 <b>وبسایت:</b>\n";
                $text .= "<a href='{$item['website_url']}'>{$item['website_url']}</a>";
                
                // ایجاد دکمه برای باز کردن سایت
                $keyboard = [
                    'inline_keyboard' => [
                        [
                            ['text' => '🌐 باز کردن سایت', 'url' => $item['website_url']]
                        ]
                    ]
                ];
                
                $this->bot->sendMessage($chatId, $text, $keyboard);
            } else {
                $this->sendTemporaryMessage($chatId, "📎 لینک وبسایت تعریف نشده");
            }
        } elseif ($section === 'news') {
            // برای اخبار: منطق پیشرفته‌تر
            $linkValue = $this->getItemLinkValue($item);
            $hasLink = !empty($linkValue);
            $hasFile = !empty($item['file_path']);
            
            // ساخت متن خبر
            $text = "📰 <b>{$item['title']}</b>\n\n";
            
            // نمایش توضیحات (description)
            if (!empty($item['description'])) {
                $text .= $item['description'] . "\n\n";
            }
            
            // نمایش بدنه خبر (body) اگر وجود دارد
            if (!empty($item['body'])) {
                $text .= $item['body'] . "\n\n";
            }
            
            // نمایش تاریخ شمسی در انتهای خبر (اگر خالی نباشد)
            if (!empty($item['persian_date']) && trim($item['persian_date']) !== '') {
                $text .= "📅 <b>تاریخ:</b> {$item['persian_date']}";
            }
            
            // اگر فقط لینک دارد: همه چیز در یک پیام
            if ($hasLink && !$hasFile) {
                $text .= "\n\n🔗 {$linkValue}";
                $this->bot->sendMessage($chatId, $text, null, null, false); // false برای نمایش preview لینک
            } 
            // اگر فایل دارد: متن در یک پیام، فایل در پیام جداگانه بدون caption
            elseif ($hasFile) {
                // ارسال متن (بدون لینک)
                $this->bot->sendMessage($chatId, $text, null, null, true);
                
                // ارسال فایل بدون caption
                $filePath = __DIR__ . '/../../' . $item['file_path'];
                if (file_exists($filePath)) {
                    $fileSize = filesize($filePath);
                    $maxFileSize = 50 * 1024 * 1024; // 50 مگابایت
                    
                    // اگر فایل بزرگ‌تر از 50 مگابایت است، لینک دانلود ارسال می‌کنیم
                    if ($fileSize > $maxFileSize) {
                        $downloadUrl = $this->getFileDownloadUrl($item['file_path']);
                        $keyboard = [
                            'inline_keyboard' => [
                                [
                                    ['text' => '📥 دانلود فایل', 'url' => $downloadUrl]
                                ]
                            ]
                        ];
                        $this->bot->sendMessage($chatId, "📥 برای دانلود فایل روی دکمه زیر کلیک کنید:", $keyboard);
                    } else {
                        // برای فایل‌های کوچک‌تر از 50 مگابایت، به صورت عادی ارسال می‌کنیم
                        // ارسال پیام موقت "در حال ارسال" (فقط برای فایل‌های بزرگ‌تر از 3 مگابایت)
                        $uploadMessages = $this->sendUploadProgressMessages($chatId, $filePath);
                        // ارسال فایل بدون caption
                        $result = $this->bot->sendDocument($chatId, $filePath, null);
                        // حذف پیام‌های موقت بعد از ارسال (موفق یا ناموفق) - فقط اگر پیامی ارسال شده باشد
                        if (!empty($uploadMessages)) {
                            $this->deleteUploadProgressMessages($chatId, $uploadMessages);
                        }
                        // در صورت خطا، پیام خطا ارسال می‌کنیم
                        if (is_array($result) && isset($result['error'])) {
                            if ($result['error'] === 'file_too_large') {
                                // اگر خطای 413 رخ داد، لینک دانلود ارسال می‌کنیم
                                $downloadUrl = $this->getFileDownloadUrl($item['file_path']);
                                $keyboard = [
                                    'inline_keyboard' => [
                                        [
                                            ['text' => '📥 دانلود فایل', 'url' => $downloadUrl]
                                        ]
                                    ]
                                ];
                                $this->bot->sendMessage($chatId, "📥 برای دانلود فایل روی دکمه زیر کلیک کنید:", $keyboard);
                            } else {
                                $this->sendTemporaryMessage($chatId, "❌ خطا در ارسال فایل");
                            }
                        } elseif (!$result) {
                            $this->sendTemporaryMessage($chatId, "❌ خطا در ارسال فایل");
                        }
                    }
                }
                
                // اگر لینک هم دارد، آن را در پیام جداگانه ارسال می‌کنیم
                if ($hasLink) {
                    $linkText = "🔗 {$linkValue}";
                    $this->bot->sendMessage($chatId, $linkText, null, null, false);
                }
            } 
            // اگر نه فایل دارد و نه لینک: فقط متن
            else {
                $this->bot->sendMessage($chatId, $text, null, null, true);
            }
        } else {
            // بررسی اینکه آیا این یک فولدر است یا فایل
            $itemType = $item['item_type'] ?? 'file';
            
            if ($itemType === 'folder') {
                // اگر فولدر است، وارد آن می‌شویم
                // contentId همان ID فولدر است که باید به عنوان parentId استفاده شود
                $this->showSection($chatId, $messageId, $section, (int)$contentId);
            } else {
                // برای فایل‌ها، ارسال لینک پیام تلگرام و/یا فایل
                $linkValue = $this->getItemLinkValue($item);
                $hasLink = !empty($linkValue);
                $hasFile = !empty($item['file_path']);
                
                if ($hasLink || $hasFile) {
                    // اگر فایل وجود دارد، ارسال فایل
                    if ($hasFile) {
                        $filePath = __DIR__ . '/../../' . $item['file_path'];
                        if (file_exists($filePath)) {
                            $fileSize = filesize($filePath);
                            $maxFileSize = 50 * 1024 * 1024; // 50 مگابایت
                            
                            // اگر فایل بزرگ‌تر از 50 مگابایت است، لینک دانلود ارسال می‌کنیم
                            if ($fileSize > $maxFileSize) {
                                $caption = $this->buildFileCaption($item, $section);
                                $downloadUrl = $this->getFileDownloadUrl($item['file_path']);
                                $keyboard = [
                                    'inline_keyboard' => [
                                        [
                                            ['text' => '📥 دانلود فایل', 'url' => $downloadUrl]
                                        ]
                                    ]
                                ];
                                $this->bot->sendMessage($chatId, $caption . "\n\n📥 برای دانلود فایل روی دکمه زیر کلیک کنید:", $keyboard);
                            } else {
                                // برای فایل‌های کوچک‌تر از 50 مگابایت، به صورت عادی ارسال می‌کنیم
                                $caption = $this->buildFileCaption($item, $section);
                                // ارسال پیام موقت "در حال ارسال" (فقط برای فایل‌های بزرگ‌تر از 3 مگابایت)
                                $uploadMessages = $this->sendUploadProgressMessages($chatId, $filePath);
                                // ارسال فایل با نام واقعی آن در سرور (بدون تغییر نام)
                                $result = $this->bot->sendDocument($chatId, $filePath, $caption);
                                // حذف پیام‌های موقت بعد از ارسال (موفق یا ناموفق) - فقط اگر پیامی ارسال شده باشد
                                if (!empty($uploadMessages)) {
                                    $this->deleteUploadProgressMessages($chatId, $uploadMessages);
                                }
                                // در صورت خطا، پیام خطا ارسال می‌کنیم
                                if (is_array($result) && isset($result['error'])) {
                                    if ($result['error'] === 'file_too_large') {
                                        // اگر خطای 413 رخ داد، لینک دانلود ارسال می‌کنیم
                                        $downloadUrl = $this->getFileDownloadUrl($item['file_path']);
                                        $keyboard = [
                                            'inline_keyboard' => [
                                                [
                                                    ['text' => '📥 دانلود فایل', 'url' => $downloadUrl]
                                                ]
                                            ]
                                        ];
                                        $this->bot->sendMessage($chatId, $caption . "\n\n📥 برای دانلود فایل روی دکمه زیر کلیک کنید:", $keyboard);
                                    } else {
                                        $this->sendTemporaryMessage($chatId, "❌ خطا در ارسال فایل");
                                    }
                                } elseif (!$result) {
                                    $this->sendTemporaryMessage($chatId, "❌ خطا در ارسال فایل");
                                }
                            }
                        } else {
                            $this->sendTemporaryMessage($chatId, "📎 فایل در سرور یافت نشد");
                        }
                    }
                    
                    // اگر لینک وجود دارد، ارسال لینک
                    if ($hasLink) {
                        $text = $this->buildLinkMessage($item, $section, $linkValue);
                        // false برای نمایش preview لینک (minimized)
                        $this->bot->sendMessage($chatId, $text, null, null, false);
                    }
                } else {
                    $this->sendTemporaryMessage($chatId, "📎 فایل تعریف نشده");
                }
            }
        }
    }
    
    // تابع parseTelegramMessageLink حذف شد چون دیگر نیازی نیست
    
    /**
     * مدیریت پیجینیشن
     */
    private function handlePagination($chatId, $messageId, $data) {
        $parts = explode('_', $data);
        $section = $parts[0];
        $page = (int)$parts[2];
        
        $sectionNames = [
            'instructions' => BotConfig::getMessage('instructions'),
            'regulations' => BotConfig::getMessage('regulations'),
            'trainings' => BotConfig::getMessage('trainings'),
            'reports' => BotConfig::getMessage('reports'),
            'news' => BotConfig::getMessage('news')
        ];
        
        $title = $sectionNames[$section] ?? $section;
        $text = "<b>{$title}</b>\n\n";
        
        // دریافت مطالب صفحه جدید
        $items = $this->models[$section]->getPaginated($page, BotConfig::MAX_RESULTS_PER_PAGE);
        $totalPages = $this->models[$section]->getTotalPages(BotConfig::MAX_RESULTS_PER_PAGE);
        
        if (empty($items)) {
            $text .= BotConfig::getMessage('no_results');
        } else {
            $text .= "📄 <b>صفحه {$page} از {$totalPages}</b>\n\n";
            foreach ($items as $index => $item) {
                $text .= "• {$item['title']}\n";
            }
        }
        
        $keyboard = KeyboardManager::getContentKeyboard($items, $section, $page, $totalPages);
        
        $this->bot->editMessageText($chatId, $messageId, $text, $keyboard);
    }
    
    /**
     * ارسال راهنما
     */
    private function sendHelp($chatId) {
        $text = "📖 <b>راهنمای استفاده</b>\n\n";
        $text .= "• از دکمه‌های زیر برای دسترسی به بخش‌های مختلف استفاده کنید\n";
        $text .= "• برای جستجو در هر بخش، روی دکمه جستجو کلیک کنید\n";
        $text .= "• برای بازگشت به منوی اصلی، از دکمه 🏠 استفاده کنید\n";
        $text .= "• برای دریافت فایل، روی عنوان مطلب کلیک کنید";
        
        $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * نمایش متن درباره ما
     */
    private function showAboutUs($chatId, $messageId = null) {
        $aboutText = $this->settingsModel->getAboutUs();
        
        if (empty($aboutText)) {
            $aboutText = "به بات جامعه مهندسین ساختمان استان قم خوش آمدید!";
        }
        
        $text = "ℹ️ <b>درباره ما</b>\n\n";
        $text .= $aboutText;
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'main_menu']
                ]
            ]
        ];
        
        // غیرفعال کردن preview برای پیام درباره ما
        if ($messageId) {
            $this->bot->editMessageText($chatId, $messageId, $text, $keyboard, true);
        } else {
            $this->bot->sendMessage($chatId, $text, $keyboard, null, true);
        }
    }
    
    /**
     * تبدیل مقدار کانال به شناسه قابل استفاده در getChatMember
     */
    private function resolveChannelChatId($channel) {
        if (empty($channel)) {
            return null;
        }

        $channel = trim($channel);
        if ($channel === '') {
            return null;
        }

        if (stripos($channel, 't.me/') !== false) {
            $channel = substr($channel, strrpos($channel, '/') + 1);
        }

        if ($channel === '') {
            return null;
        }

        if ($channel[0] !== '@' && !preg_match('/^-?\d+$/', $channel)) {
            $channel = '@' . $channel;
        }

        return $channel;
    }

    /**
     * بررسی عضویت کاربر در کانال تلگرام
     */
    private function checkChannelMembership($userId) {
        if (!$this->channelChatId || empty($userId)) {
            return true;
        }

        try {
            $response = $this->bot->getChatMember($this->channelChatId, $userId);
        } catch (Exception $e) {
            error_log('Telegram getChatMember exception: ' . $e->getMessage());
            return false;
        }

        if (empty($response) || !is_array($response)) {
            error_log('Telegram getChatMember empty response for user ' . $userId);
            return false;
        }

        if (isset($response['ok']) && $response['ok'] === true && isset($response['result']['status'])) {
            $status = $response['result']['status'];
            if (in_array($status, ['creator', 'administrator', 'member'], true)) {
                return true;
            }
            if ($status === 'restricted' && !empty($response['result']['is_member'])) {
                return true;
            }
            return false;
        }

        if (isset($response['error_code']) && in_array((int)$response['error_code'], [400, 403], true)) {
            error_log('Telegram getChatMember returned error ' . $response['error_code'] . ' for user ' . $userId . ': ' . ($response['description'] ?? ''));
            return false;
        }

        error_log('Telegram getChatMember unexpected response: ' . json_encode($response, JSON_UNESCAPED_UNICODE));
        return false;
    }

    /**
     * ارسال پیام عضویت در کانال
     */
    private function sendChannelJoinPrompt($chatId) {
        $joinUrl = BotConfig::TELEGRAM_CHANNEL;
        $text = BotConfig::getMessage('channel_required_title') . "\n\n" . BotConfig::getMessage('channel_required_subtitle');

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔔 عضویت در کانال', 'url' => $joinUrl]
                ],
                [
                    ['text' => '✅ بررسی عضویت', 'callback_data' => 'check_membership']
                ]
            ]
        ];

        $this->bot->sendMessage($chatId, $text, $keyboard, null, true);
    }

    /**
     * اطمینان از عضویت کاربر
     */
    private function ensureChannelMembership($chatId, $userId, $sendPrompt = true) {
        $isMember = $this->checkChannelMembership($userId);
        if ($isMember) {
            return true;
        }

        if ($sendPrompt) {
            $this->sendChannelJoinPrompt($chatId);
        }

        return false;
    }

    /**
     * دریافت عنوان مناسب برای نمایش
     */
    private function getItemTitle($item, $section) {
        if ($section === 'faq') {
            return $item['question'] ?? '';
        }
        if ($section === 'ethics') {
            return $item['quote_text'] ?? '';
        }
        return $item['title'] ?? '';
    }

    /**
     * دریافت لینک مرتبط با آیتم (حمایت از فرم‌ها)
     */
    private function getItemLinkValue($item) {
        if (!empty($item['telegram_message_link'])) {
            return $item['telegram_message_link'];
        }
        if (!empty($item['form_url'])) {
            return $item['form_url'];
        }
        return null;
    }
    
    /**
     * ساخت URL دانلود فایل
     */
    private function getFileDownloadUrl($filePath) {
        $baseUrl = BotConfig::getBaseUrl();
        // حذف اسلش اولیه اگر وجود دارد
        $filePath = ltrim($filePath, '/');
        return rtrim($baseUrl, '/') . '/' . $filePath;
    }

    /**
     * Escape متن برای ParseMode=HTML
     */
    private function escapeForTelegram($text) {
        return htmlspecialchars($text ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }

    /**
     * ساخت کپشن فایل
     */
    private function buildFileCaption($item, $section) {
        $title = $this->escapeForTelegram($this->getItemTitle($item, $section));
        if ($title === '') {
            $title = 'بدون عنوان';
        }

        $caption = "<b>{$title}</b>";

        return $caption;
    }

    /**
     * ساخت پیام حاوی لینک
     */
    private function buildLinkMessage($item, $section, $link) {
        $title = $this->escapeForTelegram($this->getItemTitle($item, $section));
        if ($title === '') {
            $title = 'بدون عنوان';
        }

        $safeLink = $this->escapeForTelegram($link);

        $text = "🔗 <b>{$title}</b>\n\n";
        $text .= "<a href=\"{$safeLink}\">{$safeLink}</a>\n\n";
        $text .= "💡 <i>برای مشاهده فایل، روی لینک بالا کلیک کنید</i>";

        return $text;
    }

    /**
     * ارسال پیام تبلیغاتی کانال و پین کردن آن
     */
    private function sendChannelPromotion($chatId) {
        $channelUrl = BotConfig::TELEGRAM_CHANNEL;
        
        $text = "📢 <b>کانال تلگرام ما</b>\n\n";
        $text .= "برای دریافت آخرین اخبار، دستورالعمل‌ها و به‌روزرسانی‌ها، به کانال تلگرام ما بپیوندید:\n\n";
        $text .= "🔗 " . $channelUrl;
        
        // ایجاد inline keyboard با دکمه لینک کانال
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '📢 عضویت در کانال', 'url' => $channelUrl]
                ]
            ]
        ];
        
        // ارسال پیام
        $result = $this->bot->sendMessage($chatId, $text, $keyboard);
        
        // پین کردن پیام اگر ارسال موفقیت‌آمیز بود
        if ($result && isset($result['result']['message_id'])) {
            $messageId = $result['result']['message_id'];
            // پین کردن پیام بدون اعلان
            $this->bot->pinChatMessage($chatId, $messageId, true);
        }
    }
    
    /**
     * ارسال پیام موقت (که بعد از 3 ثانیه پاک می‌شود)
     */
    private function sendTemporaryMessage($chatId, $text) {
        $result = $this->bot->sendMessage($chatId, $text);
        
        if ($result && isset($result['result']['message_id'])) {
            $messageId = $result['result']['message_id'];
            
            // ثبت لاگ برای حذف بعدی
            error_log("Message will be deleted after 3 seconds: $messageId");
            
            // اجرای حذف در پس‌زمینه بدون افکت
            $this->deleteMessageAfterDelay($chatId, $messageId, 3);
        }
    }
    
    /**
     * حذف پیام با افکت پودر بعد از تأخیر
     */
    private function deleteMessageWithEffectAfterDelay($chatId, $messageId, $delaySeconds) {
        // ایجاد یک فایل PHP موقت برای اجرای افکت
        $scriptContent = "<?php
require_once __DIR__ . '/src/TelegramBot/TelegramBot.php';
require_once __DIR__ . '/config/bot.php';

\$chatId = \$argv[1] ?? null;
\$messageId = \$argv[2] ?? null;
\$delaySeconds = \$argv[3] ?? 3;

if (!\$chatId || !\$messageId) {
    exit(1);
}

sleep(\$delaySeconds);

\$bot = new TelegramBot();

// ارسال پیام موقت با افکت پودر
\$effectEmojis = ['✨', '💫', '🌟', '💥', '🔥'];
\$randomEmoji = \$effectEmojis[array_rand(\$effectEmojis)];
\$effectText = \$randomEmoji . ' ' . \$randomEmoji . ' ' . \$randomEmoji;

// ارسال پیام افکت
\$result = \$bot->sendRequest('sendMessage', [
    'chat_id' => \$chatId,
    'text' => \$effectText,
    'reply_to_message_id' => \$messageId,
    'disable_notification' => true
]);

// حذف پیام اصلی
\$bot->deleteMessage(\$chatId, \$messageId);

// حذف پیام افکت بعد از 0.5 ثانیه
if (\$result && isset(\$result['result']['message_id'])) {
    \$effectMessageId = \$result['result']['message_id'];
    sleep(0.5);
    \$bot->deleteMessage(\$chatId, \$effectMessageId);
}
";
        
        $scriptPath = __DIR__ . "/../../delete_message_with_effect.php";
        file_put_contents($scriptPath, $scriptContent);
        
        $command = "php {$scriptPath} {$chatId} {$messageId} {$delaySeconds} > /dev/null 2>&1 &";
        
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            // برای ویندوز
            pclose(popen("start /B php {$scriptPath} {$chatId} {$messageId} {$delaySeconds}", "r"));
        } else {
            // برای لینوکس
            exec($command);
        }
    }
    
    /**
     * حذف پیام بعد از تأخیر
     */
    private function deleteMessageAfterDelay($chatId, $messageId, $delaySeconds) {
        // برای محیط عادی، از exec استفاده می‌کنیم
        $script = __DIR__ . "/../../delete_message.php";
        $command = "php {$script} {$chatId} {$messageId} {$delaySeconds} > /dev/null 2>&1 &";
        
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            // برای ویندوز
            pclose(popen("start /B php {$script} {$chatId} {$messageId} {$delaySeconds}", "r"));
        } else {
            // برای لینوکس
            exec($command);
        }
    }
    
    /**
     * ارسال یک پیام موقت "در حال ارسال" (فقط برای فایل‌های بزرگ‌تر از 3 مگابایت)
     */
    private function sendUploadProgressMessages($chatId, $filePath = null) {
        // بررسی حجم فایل - فقط برای فایل‌های بزرگ‌تر از 3 مگابایت
        if ($filePath && file_exists($filePath)) {
            $fileSize = filesize($filePath);
            $sizeInMB = $fileSize / (1024 * 1024); // تبدیل به مگابایت
            
            // اگر فایل کوچکتر از 3 مگابایت است، پیام ارسال نمی‌کنیم
            if ($sizeInMB < 3) {
                return [];
            }
        }
        
        $result = $this->bot->sendMessage($chatId, "⏳ فایل در حال ارسال است...");
        if ($result && isset($result['result']['message_id'])) {
            return [$result['result']['message_id']];
        }
        return [];
    }
    
    /**
     * حذف پیام موقت "در حال ارسال"
     */
    private function deleteUploadProgressMessages($chatId, $messageIds) {
        if (empty($messageIds)) {
            return;
        }
        
        // حذف اولین پیام (تنها پیام) بدون افکت
        if (isset($messageIds[0])) {
            $this->bot->deleteMessage($chatId, $messageIds[0]);
        }
    }
    
    /**
     * حذف پیام با افکت پودر شدن
     */
    private function deleteMessageWithEffect($chatId, $messageId) {
        // ارسال پیام موقت با افکت پودر
        $effectEmojis = ['✨', '💫', '🌟', '💥', '🔥'];
        $randomEmoji = $effectEmojis[array_rand($effectEmojis)];
        $effectText = $randomEmoji . ' ' . $randomEmoji . ' ' . $randomEmoji;
        
        // ارسال پیام افکت در همان موقعیت (reply به پیام اصلی)
        $result = $this->bot->sendRequest('sendMessage', [
            'chat_id' => $chatId,
            'text' => $effectText,
            'reply_to_message_id' => $messageId,
            'disable_notification' => true
        ]);
        
        // حذف پیام اصلی
        $this->bot->deleteMessage($chatId, $messageId);
        
        // حذف پیام افکت بعد از 0.5 ثانیه
        if ($result && isset($result['result']['message_id'])) {
            $effectMessageId = $result['result']['message_id'];
            // استفاده از deleteMessageAfterDelay برای حذف پیام افکت
            $this->deleteMessageAfterDelay($chatId, $effectMessageId, 0.5);
        }
    }
}