<?php
// تنظیم encoding به UTF-8 برای پشتیبانی از کاراکترهای فارسی
if (function_exists('mb_internal_encoding')) {
    mb_internal_encoding('UTF-8');
}
if (function_exists('mb_regex_encoding')) {
    mb_regex_encoding('UTF-8');
}
if (function_exists('mb_http_output')) {
    mb_http_output('UTF-8');
}

require_once __DIR__ . '/auth.php';
require_login();

require_once __DIR__ . '/../src/Database/Database.php';
require_once __DIR__ . '/../src/Models/AdminModel.php';
require_once __DIR__ . '/../src/Models/AllModels.php';
require_once __DIR__ . '/../src/Models/BotUserModel.php';
require_once __DIR__ . '/helpers/date_helper.php';

$currentUser = getCurrentUser();
$adminModel = new AdminModel();
$settingsModel = new SettingsModel();
$botUserModel = new BotUserModel();

// پردازش درخواست‌ها
$action = $_POST['action'] ?? $_GET['action'] ?? '';
$section = $_POST['section'] ?? $_GET['section'] ?? '';
$parentId = isset($_GET['parent_id']) ? (int)$_GET['parent_id'] : null; // ID فولدر والد برای نمایش محتویات

$models = [
    'instructions' => new InstructionModel(),
    'regulations' => new RegulationModel(),
    'trainings' => new TrainingModel(),
    'reports' => new ReportModel(),
    'news' => new NewsModel(),
    'faq' => new FaqModel(),
    'ai' => new AiModel(),
    'ethics' => new EthicModel(),
    'forms' => new FormModel()
];
$broadcastModel = new BroadcastMessageModel();
$tagModel = new ContentTagModel();

$sectionNames = [
    'instructions' => 'دستورالعمل‌ها',
    'regulations' => 'آیین‌نامه‌ها',
    'trainings' => 'آموزش‌ها',
    'reports' => 'گزارش‌نویسی',
    'news' => 'اخبار',
    'faq' => 'سوالات متداول',
    'ai' => 'دستیارهای هوشمند',
    'ethics' => 'اخلاق مهندسی',
    'forms' => 'فرم‌ها'
];

$fileBasedSections = ['instructions', 'regulations', 'trainings', 'reports', 'news', 'forms'];
$simpleTextSections = ['faq', 'ai', 'ethics'];

if (!function_exists('parseTagsInput')) {
    function parseTagsInput($input) {
        $tags = preg_split('/[\n,]+/u', (string)$input);
        $clean = [];
        foreach ($tags as $tag) {
            $tag = trim($tag);
            if ($tag === '') {
                continue;
            }
            if (mb_strlen($tag) > 50) {
                $tag = mb_substr($tag, 0, 50);
            }
            if (!in_array($tag, $clean, true)) {
                $clean[] = $tag;
            }
            if (count($clean) >= 10) {
                break;
            }
        }
        return $clean;
    }
}

if (!function_exists('formatFileSize')) {
    function formatFileSize($bytes) {
        if (!is_numeric($bytes) || $bytes <= 0) {
            return 'نامشخص';
        }
        $units = ['بایت', 'کیلوبایت', 'مگابایت', 'گیگابایت', 'ترابایت'];
        $power = (int)floor(log($bytes, 1024));
        $power = min($power, count($units) - 1);
        $value = $bytes / (1024 ** $power);
        if ($power === 0) {
            return number_format($value, 0) . ' ' . $units[$power];
        }
        return number_format($value, 2) . ' ' . $units[$power];
    }
}

if (!function_exists('setLinkFieldValue')) {
    function setLinkFieldValue(&$data, $section, $linkValue) {
        if ($section === 'forms') {
            $data['form_url'] = $linkValue !== '' ? $linkValue : null;
            unset($data['telegram_message_link']);
        } else {
            $data['telegram_message_link'] = $linkValue !== '' ? $linkValue : null;
            if (isset($data['form_url'])) {
                unset($data['form_url']);
            }
        }
    }
}

if (!function_exists('clearLinkFieldValue')) {
    function clearLinkFieldValue(&$data, $section) {
        if ($section === 'forms') {
            $data['form_url'] = null;
        } else {
            $data['telegram_message_link'] = null;
        }
    }
}

if (!function_exists('getItemLinkValue')) {
    function getItemLinkValue($item, $section) {
        if ($section === 'forms') {
            return $item['form_url'] ?? '';
        }
        return $item['telegram_message_link'] ?? '';
    }
}

if (!function_exists('startBroadcastSending')) {
    function startBroadcastSending($broadcastId) {
        $broadcastId = (int)$broadcastId;
        if ($broadcastId <= 0) {
            return false;
        }

        $script = __DIR__ . '/../scripts/send_broadcast.php';
        if (!file_exists($script)) {
            error_log("Broadcast script not found at {$script}");
            return false;
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $command = 'start /B "" php "' . $script . '" ' . $broadcastId;
            pclose(popen($command, 'r'));
        } else {
            $command = 'php ' . escapeshellarg($script) . ' ' . $broadcastId . ' > /dev/null 2>&1 &';
            exec($command);
        }

        return true;
    }
}

$success_message = '';
$error_message = '';
$maxUploadSize = 128 * 1024 * 1024; // 128MB

// پردازش عملیات
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($action) {
        case 'add':
            $title = $_POST['title'] ?? '';
            $telegram_message_link = $_POST['telegram_message_link'] ?? '';
            $tagsInput = $section === 'ethics' ? '' : ($_POST['tags'] ?? '');
            $tagsList = $section === 'ethics' ? [] : parseTagsInput($tagsInput);
            
            // بررسی فیلدهای الزامی
            $requiredFields = [];
            if ($section === 'faq') {
                if (empty($_POST['question'])) $requiredFields[] = 'سؤال';
                if (empty($_POST['answer'])) $requiredFields[] = 'پاسخ';
            } elseif ($section === 'ai') {
                if (empty($_POST['title'])) $requiredFields[] = 'عنوان';
            } elseif ($section === 'ethics') {
                if (empty($_POST['quote_text'])) $requiredFields[] = 'جمله';
            } elseif ($section === 'forms') {
                if (empty($_POST['title'])) $requiredFields[] = 'عنوان';
            } else {
                if (empty($_POST['title'])) $requiredFields[] = 'عنوان';
            }
            
            // برای بخش‌های دارای فایل، بررسی می‌کنیم که حداقل یکی از لینک تلگرام یا فایل پر شده باشد
            if (!empty($requiredFields)) {
                $error_message = "لطفاً فیلدهای زیر را پر کنید: " . implode(', ', $requiredFields);
            } elseif (in_array($section, $fileBasedSections, true)) {
                // بررسی اینکه برای فایل‌ها (نه فولدرها) یکی از لینک تلگرام یا فایل باید پر شده باشد
                $itemType = $_POST['item_type'] ?? 'file';
                if ($itemType === 'file') {
                    $contentType = $_POST['content_type'] ?? 'link';
                    $hasLink = !empty($_POST['telegram_message_link']);
                    $hasFile = isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK;
                    
                    if ($contentType === 'link' && !$hasLink) {
                        $error_message = "لطفاً لینک را وارد کنید";
                    } elseif ($contentType === 'file' && !$hasFile) {
                        $error_message = "لطفاً یک فایل انتخاب کنید";
                    }
                }
            }
            
            if (empty($error_message)) {
                $data = [
                    'title' => $_POST['title'] ?? '',
                    'created_by' => $currentUser['id']
                ];
                
                if ($section === 'faq') {
                    $data = [
                        'question' => $_POST['question'] ?? '',
                        'answer' => $_POST['answer'] ?? '',
                        'created_by' => $currentUser['id']
                    ];
                } elseif ($section === 'ai') {
                    $data = [
                        'title' => $_POST['title'] ?? '',
                        'website_url' => $_POST['website_url'] ?? '',
                        'created_by' => $currentUser['id']
                    ];
                } elseif ($section === 'ethics') {
                    $data = [
                        'quote_text' => $_POST['quote_text'] ?? '',
                        'created_by' => $currentUser['id']
                    ];
                } elseif ($section === 'forms') {
                    $data = [
                        'title' => $_POST['title'] ?? '',
                        'created_by' => $currentUser['id']
                    ];
                } else {
                    // برای سایر بخش‌ها (instructions, regulations, trainings, reports, news, forms)
                    $data = [
                        'title' => $_POST['title'] ?? '',
                        'created_by' => $currentUser['id']
                    ];
                    
                    // برای اخبار: افزودن body و persian_date
                    if ($section === 'news') {
                        $data['body'] = $_POST['body'] ?? '';
                        $data['persian_date'] = $_POST['persian_date'] ?? '';
                    }
                    
                    // بر اساس content_type (فقط برای add)
                    // در edit، content_type وجود ندارد، پس همان منطق قبلی را حفظ می‌کنیم
                    if (isset($_POST['content_type'])) {
                        $contentType = $_POST['content_type'];
                        if ($contentType === 'link') {
                            setLinkFieldValue($data, $section, $telegram_message_link);
                            unset($data['file_path']);
                        } else {
                            clearLinkFieldValue($data, $section);
                        }
                    } else {
                        // برای edit
                        setLinkFieldValue($data, $section, $telegram_message_link);
                    }
                }
                
                // افزودن is_pinned برای همه بخش‌ها
                if (isset($_POST['is_pinned'])) {
                    $data['is_pinned'] = ($_POST['is_pinned'] === '1' || $_POST['is_pinned'] === 'on') ? 1 : 0;
                }
                
                // اضافه کردن فیلدهای جدید برای همه بخش‌ها به جز FAQ و AI
                if (in_array($section, $fileBasedSections, true)) {
                    $data['item_type'] = $_POST['item_type'] ?? 'file'; // folder یا file
                    // اگر default_parent_id وجود دارد، از آن استفاده می‌کنیم (برای افزودن در فولدر فعلی)
                    if (!empty($_POST['default_parent_id'])) {
                        $parentIdValue = (int)$_POST['default_parent_id'];
                        // بررسی وجود parent در دیتابیس
                        $parentExists = $models[$section]->getById($parentIdValue);
                        $data['parent_id'] = ($parentExists && is_array($parentExists)) ? $parentIdValue : null;
                    } else {
                        $parentIdValue = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : 0;
                        // اگر parent_id خالی یا 0 است، باید null باشد
                        if ($parentIdValue > 0) {
                            // بررسی وجود parent در دیتابیس
                            $parentExists = $models[$section]->getById($parentIdValue);
                            $data['parent_id'] = ($parentExists && is_array($parentExists)) ? $parentIdValue : null;
                        } else {
                            $data['parent_id'] = null;
                        }
                    }
                    // ترتیب نمایش به صورت خودکار تنظیم می‌شود (بر اساس آخرین id)
                    $data['display_order'] = 0; // مقدار پیش‌فرض
                    
                    // اگر فولدر است، نیازی به لینک یا فایل نیست
                    if ($data['item_type'] === 'folder') {
                        clearLinkFieldValue($data, $section);
                        unset($data['file_path']);
                    }
                }
                
                // آپلود فایل (فقط برای بخش‌هایی که می‌توانند فایل داشته باشند و content_type = file)
                $contentType = $_POST['content_type'] ?? 'link';
                if (in_array($section, $fileBasedSections, true) && $contentType === 'file' && isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
                    if ($_FILES['file']['size'] > $maxUploadSize) {
                        $error_message = "حجم فایل نباید بیشتر از ۱۲۸ مگابایت باشد.";
                    } else {
                        $uploadDir = __DIR__ . '/../uploads/';
                        if (!file_exists($uploadDir)) {
                            mkdir($uploadDir, 0755, true);
                        }
                        
                        $originalName = $_FILES['file']['name'];
                        $fileExtension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
                        
                        // اولویت با فیلد file_name است، سپس title، سپس نام فایل اصلی
                        $customFileName = isset($_POST['file_name']) ? trim($_POST['file_name']) : '';
                        // استفاده مستقیم از $_POST برای title تا مطمئن شویم که مقدار دارد
                        $title = isset($_POST['title']) ? trim($_POST['title']) : '';
                        $originalBaseName = pathinfo($originalName, PATHINFO_FILENAME);
                        
                        // لاگ برای دیباگ (با encoding درست)
                        error_log("File upload - POST title (raw): '" . ($_POST['title'] ?? 'NULL') . "', POST title (trimmed): '{$title}', POST file_name (raw): '" . ($_POST['file_name'] ?? 'NULL') . "', customFileName (trimmed): '{$customFileName}', originalName: '{$originalName}', originalBaseName: '{$originalBaseName}'");
                        error_log("File upload - title length: " . mb_strlen($title) . ", customFileName length: " . mb_strlen($customFileName) . ", title empty: " . (empty($title) ? 'YES' : 'NO') . ", customFileName empty: " . (empty($customFileName) ? 'YES' : 'NO'));
                        
                        // تعیین نام پایه برای فایل
                        $baseName = null;
                        $source = 'none';
                        
                        if (!empty($customFileName)) {
                            // اگر نام فایل به صورت دستی وارد شده، از آن استفاده می‌کنیم
                            $baseName = $customFileName;
                            $source = 'custom';
                            error_log("Using custom file name: {$baseName}");
                        } elseif (!empty($title)) {
                            // در غیر این صورت، از title استفاده می‌کنیم
                            $baseName = $title;
                            $source = 'title';
                            error_log("Using title as file name: {$baseName}");
                        } elseif (!empty($originalBaseName)) {
                            // در غیر این صورت، از نام فایل اصلی استفاده می‌کنیم
                            $baseName = $originalBaseName;
                            $source = 'original';
                            error_log("Using original file name: {$baseName}");
                        } else {
                            // آخرین حالت: استفاده از 'file'
                            $baseName = 'file';
                            $source = 'default';
                            error_log("Using default file name: {$baseName}");
                        }
                        
                        // پاکسازی نام فایل (پشتیبانی کامل از فارسی)
                        // نگه داشتن تمام حروف (شامل فارسی)، اعداد، خط تیره، زیرخط و فاصله
                        // فقط کاراکترهای خطرناک برای نام فایل را حذف می‌کنیم: / \ : * ? " < > |
                        $safeBaseName = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $baseName);
                        // تبدیل فاصله‌های متوالی به یک زیرخط (اختیاری - می‌توانیم فاصله‌ها را نگه داریم)
                        // $safeBaseName = preg_replace('/\s+/u', '_', $safeBaseName);
                        // حذف فاصله‌ها و زیرخط‌های اضافی از ابتدا و انتها
                        $safeBaseName = trim($safeBaseName);
                        
                        error_log("After cleanup - baseName: '{$baseName}', safeBaseName: '{$safeBaseName}', source: {$source}, length: " . mb_strlen($safeBaseName));
                        
                        // بررسی اینکه آیا نام فایل خالی است یا نه (با استفاده از mb_strlen برای پشتیبانی از UTF-8)
                        $isEmpty = mb_strlen($safeBaseName) === 0;
                        
                        // اگر بعد از پاکسازی خالی شد و از title یا customFileName استفاده کرده بودیم، از نام فایل اصلی استفاده می‌کنیم
                        if ($isEmpty && ($source === 'title' || $source === 'custom')) {
                            if (!empty($originalBaseName)) {
                                $safeBaseName = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $originalBaseName);
                                $safeBaseName = trim($safeBaseName);
                                $isEmpty = mb_strlen($safeBaseName) === 0;
                                error_log("Cleanup made it empty, using original: '{$safeBaseName}'");
                            }
                        }
                        
                        // اگر هنوز خالی است، از نام پیش‌فرض استفاده می‌کنیم
                        if ($isEmpty) {
                            $safeBaseName = 'file';
                            error_log("Still empty after all attempts, using default: '{$safeBaseName}'");
                        }
                        
                        error_log("Final safe file name: '{$safeBaseName}' (source: {$source}, from: '{$baseName}', length: " . mb_strlen($safeBaseName) . ")");
                        
                        // محدود کردن طول نام فایل
                        if (mb_strlen($safeBaseName) > 100) {
                            $safeBaseName = mb_substr($safeBaseName, 0, 100);
                        }
                        
                        // ساخت نام فایل نهایی با پسوند
                        $fileName = $safeBaseName;
                        if ($fileExtension !== '') {
                            $fileName .= '.' . $fileExtension;
                        }
                        
                        // اگر فایل با نام جدید وجود دارد، یک نام منحصر به فرد ایجاد می‌کنیم
                        $filePath = $uploadDir . $fileName;
                        $counter = 1;
                        while (file_exists($filePath)) {
                            $fileName = $safeBaseName . '_' . $counter;
                            if ($fileExtension !== '') {
                                $fileName .= '.' . $fileExtension;
                            }
                            $filePath = $uploadDir . $fileName;
                            $counter++;
                        }
                        
                        // آپلود فایل با نام جدید
                        if (move_uploaded_file($_FILES['file']['tmp_name'], $filePath)) {
                            $data['file_path'] = 'uploads/' . $fileName;
                            error_log("New file uploaded with name: {$fileName} (based on: " . (!empty($customFileName) ? 'file_name' : (!empty($title) ? 'title' : 'original')) . ")");
                        } else {
                            error_log("Failed to upload file: {$filePath}");
                        }
                    }
                }
                
                if (empty($error_message)) {
                    $newId = $models[$section]->create($data);
                    if ($newId) {
                        $tagModel->setTags($section, $newId, $tagsList);
                        logAction('create', "افزودن مطلب جدید در بخش {$sectionNames[$section]}");
                        $success_message = "مطلب با موفقیت افزوده شد";
                        
                        // بررسی اینکه آیا درخواست AJAX است یا نه
                        $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
                        
                        if ($isAjax) {
                            // پاسخ JSON برای AJAX
                            header('Content-Type: application/json; charset=utf-8');
                            echo json_encode([
                                'success' => true,
                                'message' => $success_message
                            ], JSON_UNESCAPED_UNICODE);
                            exit;
                        }
                    } else {
                        $error_message = "خطا در افزودن مطلب";
                        
                        // بررسی اینکه آیا درخواست AJAX است یا نه
                        $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
                        
                        if ($isAjax) {
                            // پاسخ JSON برای AJAX
                            header('Content-Type: application/json; charset=utf-8');
                            echo json_encode([
                                'success' => false,
                                'message' => $error_message
                            ], JSON_UNESCAPED_UNICODE);
                            exit;
                        }
                    }
                }
            }
            break;
            
        case 'edit':
            $id = $_POST['id'] ?? 0;
            
            if (empty($id) || empty($section)) {
                $error_message = "اطلاعات ناقص است";
            } else {
                // بررسی دسترسی
                $item = $models[$section]->getById($id);
                if (!$adminModel->canModifyContent($currentUser['id'], $item['created_by'], $currentUser['role'])) {
                    $error_message = "شما دسترسی ویرایش این مطلب را ندارید";
                } else {
                    $tagsInput = $section === 'ethics' ? '' : ($_POST['tags'] ?? '');
                    $tagsList = $section === 'ethics' ? [] : parseTagsInput($tagsInput);
                    $data = [];
            
            if ($section === 'faq') {
                        $data = [
                            'question' => $_POST['question'] ?? '',
                            'answer' => $_POST['answer'] ?? ''
                        ];
                    } elseif ($section === 'ai') {
                        $data = [
                            'title' => $_POST['title'] ?? '',
                            'website_url' => $_POST['website_url'] ?? ''
                        ];
                    } elseif ($section === 'ethics') {
                        $data = [
                            'quote_text' => $_POST['quote_text'] ?? ''
                        ];
                    } elseif ($section === 'forms') {
                        $data = [
                            'title' => $_POST['title'] ?? ''
                        ];
            } else {
                        $data = [
                            'title' => $_POST['title'] ?? '',
                            'telegram_message_link' => $_POST['telegram_message_link'] ?? ''
                        ];
                        // برای اخبار: افزودن body و persian_date
                        if ($section === 'news') {
                            $data['body'] = $_POST['body'] ?? '';
                            $data['persian_date'] = $_POST['persian_date'] ?? '';
                        }
                    }
                    
                    // بررسی فیلدهای الزامی
                    $requiredFields = [];
                    if ($section === 'faq') {
                        if (empty($data['question'])) $requiredFields[] = 'سؤال';
                        if (empty($data['answer'])) $requiredFields[] = 'پاسخ';
                    } elseif ($section === 'ai') {
                        if (empty($data['title'])) $requiredFields[] = 'عنوان';
                    } elseif ($section === 'ethics') {
                        if (empty($data['quote_text'])) $requiredFields[] = 'جمله';
                    } elseif ($section === 'forms') {
                        if (empty($data['title'])) $requiredFields[] = 'عنوان';
            } else {
                        if (empty($data['title'])) $requiredFields[] = 'عنوان';
                    }
                    
                    if (!empty($requiredFields)) {
                        $error_message = "لطفاً فیلدهای زیر را پر کنید: " . implode(', ', $requiredFields);
                    } elseif (in_array($section, $fileBasedSections, true)) {
                        // بررسی اینکه برای فایل‌ها (نه فولدرها) یکی از لینک تلگرام یا فایل باید وجود داشته باشد
                        $itemType = $_POST['item_type'] ?? ($item['item_type'] ?? 'file');
                        if ($itemType === 'file') {
                            $contentType = $_POST['content_type'] ?? (empty($item['file_path']) ? 'link' : 'file');
                            $hasLink = !empty($_POST['telegram_message_link']);
                            $hasNewFile = isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK;
                            $hasOldFile = !empty($item['file_path']);
                            
                            if ($contentType === 'link' && !$hasLink) {
                                $error_message = "لطفاً لینک را وارد کنید";
                            } elseif ($contentType === 'file' && !$hasNewFile && !$hasOldFile) {
                                $error_message = "لطفاً یک فایل انتخاب کنید یا فایل قبلی را نگه دارید";
                            }
                        }
                    }
                    
                    if (empty($error_message)) {
                        // آپلود فایل جدید (اگر آپلود شده)
                        if (in_array($section, $fileBasedSections, true) && isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
                            if ($_FILES['file']['size'] > $maxUploadSize) {
                                $error_message = "حجم فایل نباید بیشتر از ۱۲۸ مگابایت باشد.";
                            } else {
                                // حذف فایل قدیمی (اگر وجود داشته باشد)
                                if (!empty($item['file_path']) && $item['file_path'] !== null) {
                                    $oldFilePath = __DIR__ . '/../' . $item['file_path'];
                                    if (file_exists($oldFilePath) && strpos($oldFilePath, 'uploads/') !== false) {
                                        @unlink($oldFilePath);
                                        error_log("Old file deleted: " . $oldFilePath);
                                    }
                                }
                                
                                $uploadDir = __DIR__ . '/../uploads/';
                                if (!file_exists($uploadDir)) {
                                    mkdir($uploadDir, 0755, true);
                                }
                                
                                $originalName = $_FILES['file']['name'];
                                $fileExtension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
                                
                                // اولویت با فیلد file_name است، سپس title، سپس نام فایل اصلی
                                $customFileName = trim($_POST['file_name'] ?? '');
                                // استفاده مستقیم از $_POST برای title تا مطمئن شویم که مقدار دارد
                                $title = trim($_POST['title'] ?? '');
                                $originalBaseName = pathinfo($originalName, PATHINFO_FILENAME);
                                
                                // لاگ برای دیباگ
                                error_log("File upload (edit) - POST title: '" . ($_POST['title'] ?? 'NULL') . "', customFileName: '{$customFileName}', title (trimmed): '{$title}', originalName: '{$originalName}', originalBaseName: '{$originalBaseName}'");
                                
                                // تعیین نام پایه برای فایل
                                $baseName = null;
                                $source = 'none';
                                
                                if (!empty($customFileName)) {
                                    // اگر نام فایل به صورت دستی وارد شده، از آن استفاده می‌کنیم
                                    $baseName = $customFileName;
                                    $source = 'custom';
                                    error_log("Using custom file name: {$baseName}");
                                } elseif (!empty($title)) {
                                    // در غیر این صورت، از title استفاده می‌کنیم
                                    $baseName = $title;
                                    $source = 'title';
                                    error_log("Using title as file name: {$baseName}");
                                } elseif (!empty($originalBaseName)) {
                                    // در غیر این صورت، از نام فایل اصلی استفاده می‌کنیم
                                    $baseName = $originalBaseName;
                                    $source = 'original';
                                    error_log("Using original file name: {$baseName}");
                                } else {
                                    // آخرین حالت: استفاده از 'file'
                                    $baseName = 'file';
                                    $source = 'default';
                                    error_log("Using default file name: {$baseName}");
                                }
                                
                                // پاکسازی نام فایل (پشتیبانی کامل از فارسی)
                                // نگه داشتن تمام حروف (شامل فارسی)، اعداد، خط تیره، زیرخط و فاصله
                                // فقط کاراکترهای خطرناک برای نام فایل را حذف می‌کنیم: / \ : * ? " < > |
                                $safeBaseName = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $baseName);
                                // تبدیل فاصله‌های متوالی به یک زیرخط (اختیاری - می‌توانیم فاصله‌ها را نگه داریم)
                                // $safeBaseName = preg_replace('/\s+/u', '_', $safeBaseName);
                                // حذف فاصله‌ها و زیرخط‌های اضافی از ابتدا و انتها
                                $safeBaseName = trim($safeBaseName);
                                
                                error_log("After cleanup (edit) - baseName: '{$baseName}', safeBaseName: '{$safeBaseName}', source: {$source}, length: " . mb_strlen($safeBaseName));
                                
                                // بررسی اینکه آیا نام فایل خالی است یا نه (با استفاده از mb_strlen برای پشتیبانی از UTF-8)
                                $isEmpty = mb_strlen($safeBaseName) === 0;
                                
                                // اگر بعد از پاکسازی خالی شد و از title یا customFileName استفاده کرده بودیم، از نام فایل اصلی استفاده می‌کنیم
                                if ($isEmpty && ($source === 'title' || $source === 'custom')) {
                                    if (!empty($originalBaseName)) {
                                        $safeBaseName = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $originalBaseName);
                                        $safeBaseName = trim($safeBaseName);
                                        $isEmpty = mb_strlen($safeBaseName) === 0;
                                        error_log("Cleanup made it empty, using original: '{$safeBaseName}'");
                                    }
                                }
                                
                                // اگر هنوز خالی است، از نام پیش‌فرض استفاده می‌کنیم
                                if ($isEmpty) {
                                    $safeBaseName = 'file';
                                    error_log("Still empty after all attempts, using default: '{$safeBaseName}'");
                                }
                                
                                error_log("Final safe file name (edit): '{$safeBaseName}' (source: {$source}, from: '{$baseName}', length: " . mb_strlen($safeBaseName) . ")");
                                
                                // محدود کردن طول نام فایل
                                if (mb_strlen($safeBaseName) > 100) {
                                    $safeBaseName = mb_substr($safeBaseName, 0, 100);
                                }
                                
                                // ساخت نام فایل نهایی با پسوند
                                $fileName = $safeBaseName;
                                if ($fileExtension !== '') {
                                    $fileName .= '.' . $fileExtension;
                                }
                                
                                // اگر فایل با نام جدید وجود دارد، یک نام منحصر به فرد ایجاد می‌کنیم
                                $filePath = $uploadDir . $fileName;
                                $counter = 1;
                                while (file_exists($filePath)) {
                                    $fileName = $safeBaseName . '_' . $counter;
                                    if ($fileExtension !== '') {
                                        $fileName .= '.' . $fileExtension;
                                    }
                                    $filePath = $uploadDir . $fileName;
                                    $counter++;
                                }
                                
                                if (move_uploaded_file($_FILES['file']['tmp_name'], $filePath)) {
                                    $data['file_path'] = 'uploads/' . $fileName;
                                }
                            }
                        }
                    }
                    
                    if (empty($error_message)) {
                        // اضافه کردن فیلدهای جدید برای همه بخش‌ها به جز FAQ و AI
                        if (in_array($section, $fileBasedSections, true)) {
                            $data['item_type'] = $_POST['item_type'] ?? 'file'; // folder یا file
                            
                            // مدیریت parent_id - باید null یا یک ID معتبر باشد
                            $parentIdValue = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : 0;
                            if ($parentIdValue > 0) {
                                // بررسی وجود parent در دیتابیس
                                $parentExists = $models[$section]->getById($parentIdValue);
                                // همچنین بررسی می‌کنیم که parent نباید خود آیتم باشد (برای جلوگیری از self-reference)
                                if ($parentExists && is_array($parentExists) && $parentIdValue != $id) {
                                    $data['parent_id'] = $parentIdValue;
                                } else {
                                    $data['parent_id'] = null;
                                }
                            } else {
                                $data['parent_id'] = null;
                            }
                            
                            // ترتیب نمایش به صورت خودکار تنظیم می‌شود (بر اساس آخرین id)
                    $data['display_order'] = 0; // مقدار پیش‌فرض
                            
                            // اگر فولدر است، نیازی به لینک یا فایل نیست
                            if ($data['item_type'] === 'folder') {
                                clearLinkFieldValue($data, $section);
                                unset($data['file_path']);
                            } else {
                                // برای فایل‌ها، بر اساس content_type تنظیم می‌کنیم
                                $hasExistingFile = !empty($item['file_path']);
                                $contentType = $_POST['content_type'] ?? ($hasExistingFile ? 'file' : 'link');
                                if ($contentType === 'link') {
                                    // اگر لینک انتخاب شده، فایل را حذف می‌کنیم (مگر اینکه فایل جدید آپلود شده باشد)
                                    if (!isset($data['file_path'])) {
                                        // حذف فایل فیزیکی قدیمی از سرور
                                        if (!empty($item['file_path']) && $item['file_path'] !== null) {
                                            $oldFilePath = __DIR__ . '/../' . $item['file_path'];
                                            if (file_exists($oldFilePath) && strpos($oldFilePath, 'uploads/') !== false) {
                                                @unlink($oldFilePath);
                                                error_log("File deleted when switching to link: " . $oldFilePath);
                                            }
                                        }
                                        $data['file_path'] = null; // حذف فایل قبلی
                                    }
                                    setLinkFieldValue($data, $section, $_POST['telegram_message_link'] ?? '');
                                } else {
                                    // اگر فایل انتخاب شده، لینک را حذف می‌کنیم
                                    clearLinkFieldValue($data, $section);
                                }
                            }
                        }
                        
                        // تغییر نام فایل فیزیکی در سرور
                        // اولویت با فیلد file_name است، در غیر این صورت از title استفاده می‌شود
                        $hasNewFile = isset($data['file_path']) && !empty($data['file_path']);
                        $shouldRenameFile = false;
                        $newFileNameBase = null;
                        
                        if (in_array($section, $fileBasedSections, true) && !empty($item['file_path']) && !$hasNewFile) {
                            // استخراج نام فایل فعلی (بدون پسوند)
                            $currentFileName = basename($item['file_path']);
                            $currentFileNameWithoutExt = pathinfo($currentFileName, PATHINFO_FILENAME);
                            
                            // بررسی اینکه آیا نام فایل باید تغییر کند
                            $customFileName = trim($_POST['file_name'] ?? '');
                            $oldTitle = $item['title'] ?? '';
                            $newTitle = $data['title'] ?? '';
                            $titleChanged = ($oldTitle !== $newTitle) && !empty($newTitle);
                            
                            // اولویت با فیلد file_name است
                            if (!empty($customFileName)) {
                                // اگر نام فایل به صورت دستی وارد شده و با نام فعلی متفاوت است
                                if ($customFileName !== $currentFileNameWithoutExt) {
                                    $shouldRenameFile = true;
                                    $newFileNameBase = $customFileName;
                                }
                            } elseif ($titleChanged) {
                                // اگر نام فایل دستی وارد نشده و title تغییر کرده، از title جدید استفاده می‌کنیم
                                // فقط کاراکترهای خطرناک را حذف می‌کنیم
                                $safeTitleForComparison = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $newTitle);
                                $safeTitleForComparison = trim($safeTitleForComparison);
                                
                                // فقط اگر title جدید با نام فایل فعلی متفاوت باشد
                                if ($safeTitleForComparison !== $currentFileNameWithoutExt && mb_strlen($safeTitleForComparison) > 0) {
                                    $shouldRenameFile = true;
                                    $newFileNameBase = $newTitle;
                                }
                            }
                            
                            // اگر باید نام فایل را تغییر دهیم
                            if ($shouldRenameFile && !empty($newFileNameBase)) {
                                $oldFilePath = __DIR__ . '/../' . $item['file_path'];
                                
                                if (file_exists($oldFilePath) && strpos($oldFilePath, 'uploads/') !== false) {
                                    // دریافت پسوند فایل
                                    $fileExtension = pathinfo($oldFilePath, PATHINFO_EXTENSION);
                                    
                                    // ایجاد نام جدید برای فایل (پشتیبانی کامل از فارسی)
                                    // فقط کاراکترهای خطرناک برای نام فایل را حذف می‌کنیم: / \ : * ? " < > |
                                    $safeFileName = preg_replace('/[\/\\\\:\*\?"<>\|]/u', '_', $newFileNameBase);
                                    $safeFileName = trim($safeFileName);
                                    
                                    // بررسی اینکه آیا نام فایل خالی است یا نه
                                    if (mb_strlen($safeFileName) === 0) {
                                        $safeFileName = 'file';
                                    }
                                    
                                    // محدود کردن طول نام فایل
                                    if (mb_strlen($safeFileName) > 100) {
                                        $safeFileName = mb_substr($safeFileName, 0, 100);
                                    }
                                    
                                    // ایجاد نام فایل جدید
                                    $newFileName = $safeFileName;
                                    if ($fileExtension !== '') {
                                        $newFileName .= '.' . $fileExtension;
                                    }
                                    
                                    $uploadDir = __DIR__ . '/../uploads/';
                                    $newFilePath = $uploadDir . $newFileName;
                                    
                                    // اگر فایل با نام جدید وجود دارد، یک نام منحصر به فرد ایجاد می‌کنیم
                                    $counter = 1;
                                    while (file_exists($newFilePath) && $newFilePath !== $oldFilePath) {
                                        $newFileName = $safeFileName . '_' . $counter;
                                        if ($fileExtension !== '') {
                                            $newFileName .= '.' . $fileExtension;
                                        }
                                        $newFilePath = $uploadDir . $newFileName;
                                        $counter++;
                                    }
                                    
                                    // تغییر نام فایل
                                    if (rename($oldFilePath, $newFilePath)) {
                                        $data['file_path'] = 'uploads/' . $newFileName;
                                        error_log("File renamed: {$oldFilePath} -> {$newFilePath}");
                                    } else {
                                        error_log("Failed to rename file: {$oldFilePath} -> {$newFilePath}");
                                    }
                                }
                            }
                        }
                        
                        if ($models[$section]->update($id, $data)) {
                            $tagModel->setTags($section, $id, $tagsList);
                            logAction('update', "ویرایش مطلب در بخش {$sectionNames[$section]}");
                            $success_message = "مطلب با موفقیت ویرایش شد";
                        } else {
                            $error_message = "خطا در ویرایش مطلب";
                        }
                    }
                }
            }
            break;
            
        case 'delete':
            $id = $_POST['id'] ?? 0;
            
            if (empty($id) || empty($section)) {
                $error_message = "اطلاعات ناقص است";
            } else {
                // بررسی دسترسی
                $item = $models[$section]->getById($id);
                if (!$adminModel->canModifyContent($currentUser['id'], $item['created_by'], $currentUser['role'])) {
                    $error_message = "شما دسترسی حذف این مطلب را ندارید";
                } else {
                    // حذف فایل فیزیکی از سرور (اگر وجود داشته باشد)
                    // فقط برای بخش‌هایی که فایل دارند (نه FAQ، نه AI، نه Forms)
                    if (in_array($section, $fileBasedSections, true)) {
                        // تابع بازگشتی برای حذف فایل‌های یک آیتم و تمام فرزندانش
                        function deleteItemFiles($model, $itemId, $section) {
                            global $tagModel;
                            $item = $model->getById($itemId);
                            if (!$item) {
                                return;
                            }
                            
                            // اگر این آیتم فایل دارد، فایل فیزیکی را حذف کن
                            if (!empty($item['file_path']) && $item['file_path'] !== null) {
                                $filePath = __DIR__ . '/../' . $item['file_path'];
                                if (file_exists($filePath) && strpos($filePath, 'uploads/') !== false) {
                                    @unlink($filePath);
                                    error_log("File deleted: " . $filePath);
                                }
                            }

                            if (isset($tagModel) && $tagModel instanceof ContentTagModel) {
                                $tagModel->setTags($section, $itemId, []);
                            }
                            
                            // پیدا کردن تمام فرزندان این آیتم (اگر فولدر باشد)
                            $allItems = $model->getAll();
                            foreach ($allItems as $childItem) {
                                $childParentId = isset($childItem['parent_id']) ? $childItem['parent_id'] : null;
                                // نرمال‌سازی parent_id برای مقایسه
                                if ($childParentId === '' || $childParentId === 0 || $childParentId === '0') {
                                    $childParentId = null;
                                } else {
                                    $childParentId = (int)$childParentId;
                                }
                                
                                if ($childParentId === (int)$itemId) {
                                    // این یک فرزند است، بازگشتی فایل‌هایش را حذف کن
                                    deleteItemFiles($model, $childItem['id'], $section);
                                }
                            }
                        }
                        
                        // حذف فایل‌های این آیتم و تمام فرزندانش
                        deleteItemFiles($models[$section], $id, $section);
                    }
                    
                    // حذف از دیتابیس
                    $tagModel->setTags($section, $id, []);
                    if ($models[$section]->delete($id)) {
                        logAction('delete', "حذف مطلب در بخش {$sectionNames[$section]}");
                        $success_message = "مطلب با موفقیت حذف شد";
                    } else {
                        $error_message = "خطا در حذف مطلب";
                    }
                }
            }
            break;

        case 'create_broadcast':
            if ($currentUser['role'] !== 'super_admin') {
                $error_message = "فقط سوپر ادمین می‌تواند پیام همگانی ارسال کند";
                break;
            }

            $messageText = trim($_POST['broadcast_message'] ?? '');
            $parseMode = ($_POST['parse_mode'] ?? 'HTML') === 'None' ? null : 'HTML';

            if (empty($messageText)) {
                $error_message = "لطفاً متن پیام همگانی را وارد کنید";
                break;
            }

            $broadcastId = $broadcastModel->createMessage([
                'message_text' => $messageText,
                'parse_mode' => $parseMode ?: null,
                'created_by' => $currentUser['id']
            ]);

            if ($broadcastId) {
                if (startBroadcastSending($broadcastId)) {
                    logAction('broadcast_create', "ارسال پیام همگانی جدید (ID: {$broadcastId})");
                    $success_message = "پیام همگانی در صف ارسال قرار گرفت";
            } else {
                    $broadcastModel->updateMessage($broadcastId, [
                        'status' => 'failed',
                        'last_error' => 'عدم موفقیت در راه‌اندازی فرایند ارسال'
                    ]);
                    $error_message = "خطا در راه‌اندازی فرآیند ارسال پیام همگانی";
                }
            } else {
                $error_message = "خطا در ذخیره پیام همگانی";
            }
            break;
            
        case 'add_admin':
            if (!canManageAdmins()) {
                $error_message = "شما دسترسی افزودن ادمین ندارید";
            } else {
                $username = $_POST['username'] ?? '';
                $password = $_POST['password'] ?? '';
                $full_name = $_POST['full_name'] ?? '';
                $role = $_POST['role'] ?? 'admin';
                
                if (empty($username) || empty($password) || empty($full_name)) {
                    $error_message = "لطفاً همه فیلدها را پر کنید";
                } else {
                    $data = [
                        'username' => $username,
                        'password' => $password,
                        'full_name' => $full_name,
                        'role' => $role,
                        'created_by' => $currentUser['id']
                    ];
                    
                    if ($adminModel->create($data)) {
                        logAction('create_admin', "افزودن ادمین جدید: {$username}");
                        $success_message = "ادمین با موفقیت افزوده شد";
                    } else {
                        $error_message = "خطا در افزودن ادمین";
                    }
                }
            }
            break;
            
        case 'edit_admin':
            if (!canManageAdmins()) {
                $error_message = "شما دسترسی مدیریت ادمین‌ها ندارید";
            } else {
                $admin_id = $_POST['admin_id'] ?? 0;
                $username = trim($_POST['username'] ?? '');
                $role = $_POST['role'] ?? 'normal';
                
                if (empty($username) || empty($admin_id)) {
                    $error_message = "لطفاً همه فیلدهای الزامی را پر کنید";
                } else {
                    // بررسی تکراری نبودن username
                    $existingAdmin = $adminModel->getByUsername($username);
                    if ($existingAdmin && $existingAdmin['id'] != $admin_id) {
                        $error_message = "این نام کاربری قبلاً استفاده شده است";
                    } else {
                        $data = [
                            'username' => $username,
                            'role' => $role
                        ];
                        
                        if ($adminModel->update($admin_id, $data)) {
                            logAction('edit_admin', "ویرایش ادمین: {$username}");
                            $success_message = "ادمین با موفقیت ویرایش شد";
                        } else {
                            $error_message = "خطا در ویرایش ادمین";
                        }
                    }
                }
            }
            break;
            
        case 'delete_admin':
            if (!canManageAdmins()) {
                $error_message = "شما دسترسی مدیریت ادمین‌ها ندارید";
            } else {
                $admin_id = $_POST['admin_id'] ?? 0;
                
                // جلوگیری از حذف خود
                if ($admin_id == $currentUser['id']) {
                    $error_message = "نمی‌توانید خودتان را حذف کنید";
                } else {
                    if ($adminModel->delete($admin_id)) {
                        logAction('delete_admin', "حذف ادمین");
                        $success_message = "ادمین با موفقیت حذف شد";
                    } else {
                        $error_message = "خطا در حذف ادمین";
                    }
                }
            }
            break;
            
        case 'change_password':
            if (!canManageAdmins()) {
                $error_message = "شما دسترسی مدیریت ادمین‌ها ندارید";
            } else {
                $admin_id = $_POST['admin_id'] ?? 0;
                $new_password = $_POST['new_password'] ?? '';
                $confirm_password = $_POST['confirm_password'] ?? '';
                
                if (empty($new_password) || empty($confirm_password) || empty($admin_id)) {
                    $error_message = "لطفاً همه فیلدهای الزامی را پر کنید";
                } elseif ($new_password !== $confirm_password) {
                    $error_message = "رمز عبور و تأیید آن مطابقت ندارند";
                } elseif (strlen($new_password) < 6) {
                    $error_message = "رمز عبور باید حداقل 6 کاراکتر باشد";
                } else {
                    if ($adminModel->changePassword($admin_id, $new_password)) {
                        logAction('change_password', "تغییر رمز عبور ادمین");
                        $success_message = "رمز عبور با موفقیت تغییر کرد";
                    } else {
                        $error_message = "خطا در تغییر رمز عبور";
                    }
                }
            }
            break;
            
        case 'change_my_password':
            // هر کاربری می‌تواند پسورد خودش را تغییر دهد
            $current_password = $_POST['current_password'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';
            
            if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                $error_message = "لطفاً همه فیلدهای الزامی را پر کنید";
            } elseif ($new_password !== $confirm_password) {
                $error_message = "رمز عبور جدید و تأیید آن مطابقت ندارند";
            } elseif (strlen($new_password) < 6) {
                $error_message = "رمز عبور جدید باید حداقل 6 کاراکتر باشد";
            } else {
                // بررسی رمز عبور فعلی
                $user = $adminModel->authenticate($currentUser['username'], $current_password);
                if (!$user) {
                    $error_message = "رمز عبور فعلی اشتباه است";
                } else {
                    if ($adminModel->changePassword($currentUser['id'], $new_password)) {
                        logAction('change_my_password', "تغییر رمز عبور خود");
                        $success_message = "رمز عبور شما با موفقیت تغییر کرد";
                    } else {
                        $error_message = "خطا در تغییر رمز عبور";
                    }
                }
            }
            break;
            
        case 'edit_about_us':
            if ($currentUser['role'] !== 'super_admin') {
                $error_message = "شما دسترسی لازم را ندارید";
            } else {
                if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                    $aboutText = $_POST['about_text'] ?? '';
                    
                    if (empty($aboutText)) {
                        $error_message = "لطفاً متن درباره ما را وارد کنید";
                    } else {
                        if ($settingsModel->setAboutUs($aboutText, $currentUser['id'])) {
                            logAction('edit_about_us', "ویرایش متن درباره ما");
                            $success_message = "متن درباره ما با موفقیت به‌روزرسانی شد";
                        } else {
                            $error_message = "خطا در به‌روزرسانی متن درباره ما";
                        }
                    }
                }
            }
            break;
            
        case 'move_order':
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $id = (int)($_POST['id'] ?? 0);
                $direction = $_POST['direction'] ?? ''; // 'up' or 'down'
            $section = $_POST['section'] ?? '';
                $parentIdParam = isset($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;
                
                if (!empty($section) && isset($models[$section]) && in_array($direction, ['up', 'down']) && $id > 0) {
                    $item = $models[$section]->getById($id);
                    if ($item && is_array($item) && $adminModel->canModifyContent($currentUser['id'], $item['created_by'], $currentUser['role'])) {
                        $currentOrder = isset($item['display_order']) ? (int)$item['display_order'] : 0;
                        $currentParentId = isset($item['parent_id']) ? $item['parent_id'] : null;
                        
                        // نرمال‌سازی parent_id برای مقایسه
                        if ($currentParentId === '' || $currentParentId === 0 || $currentParentId === '0') {
                            $currentParentId = null;
                        } elseif ($currentParentId !== null) {
                            $currentParentId = (int)$currentParentId;
                        }
                        
                        // پیدا کردن آیتم بعدی/قبلی - فقط در همان سطح (parent_id یکسان)
                        // همه ادمین‌ها می‌توانند همه محتواها را ببینند
                        $allItems = $models[$section]->getAllWithCreator();
                        
                        // فیلتر کردن بر اساس parent_id (همان سطح)
                        $sameLevelItems = array_filter($allItems, function($it) use ($currentParentId) {
                            $itParentId = isset($it['parent_id']) ? $it['parent_id'] : null;
                            // نرمال‌سازی parent_id برای مقایسه
                            if ($itParentId === '' || $itParentId === 0 || $itParentId === '0') {
                                $itParentId = null;
                            } elseif ($itParentId !== null) {
                                $itParentId = (int)$itParentId;
                            }
                            
                            // مقایسه دقیق parent_id
                            if ($currentParentId === null) {
                                return $itParentId === null;
                            }
                            return $itParentId !== null && $itParentId === $currentParentId;
                        });
                        
                        // تبدیل به آرایه عددی برای index صحیح
                        $sameLevelItems = array_values($sameLevelItems);
                        
                        // لاگ برای دیباگ - نمایش تمام آیتم‌ها
                        $debugItems = [];
                        foreach ($allItems as $it) {
                            $itParentId = isset($it['parent_id']) ? $it['parent_id'] : null;
                            if ($itParentId === '' || $itParentId === 0 || $itParentId === '0') {
                                $itParentId = null;
                            }
                            $debugItems[] = "id={$it['id']}, parent_id=" . ($itParentId !== null ? $itParentId : 'null');
                        }
                        error_log("Move order filter: allItems count=" . count($allItems) . " [" . implode(', ', $debugItems) . "], sameLevelItems count=" . count($sameLevelItems) . ", currentParentId=" . ($currentParentId !== null ? $currentParentId : 'null') . ", itemId={$id}");
                        
                        // اگر فقط یک آیتم داریم، نمی‌توان جابجا کرد
                        if (count($sameLevelItems) <= 1) {
                            $error_message = "نمی‌توان ترتیب را تغییر داد (تنها یک آیتم در این سطح وجود دارد - تعداد آیتم‌های هم‌سطح: " . count($sameLevelItems) . ")";
                        } else {
                            // مرتب‌سازی بر اساس display_order (صعودی)
                            usort($sameLevelItems, function($a, $b) {
                                $orderA = (int)($a['display_order'] ?? 0);
                                $orderB = (int)($b['display_order'] ?? 0);
                                if ($orderA === $orderB) {
                                    // اگر display_order یکسان است، بر اساس id مرتب می‌کنیم
                                    return (int)$a['id'] - (int)$b['id'];
                                }
                                return $orderA - $orderB;
                            });
                            
                            // لاگ برای دیباگ
                            error_log("Move order debug: section={$section}, id={$id}, direction={$direction}, sameLevelItems count=" . count($sameLevelItems) . ", currentParentId=" . ($currentParentId ?? 'null') . ", currentOrder={$currentOrder}");
                            
                            // پیدا کردن index آیتم فعلی
                            $currentIndex = -1;
                            foreach ($sameLevelItems as $index => $it) {
                                if ((int)$it['id'] === $id) {
                                    $currentIndex = $index;
                                    break;
                                }
                            }
                            
                            // لاگ برای دیباگ
                            $debugInfo = [];
                            foreach ($sameLevelItems as $idx => $it) {
                                $order = isset($it['display_order']) ? $it['display_order'] : 0;
                                $debugInfo[] = "id={$it['id']}, order={$order}";
                            }
                            error_log("Move order items: " . implode(', ', $debugInfo) . " | currentIndex={$currentIndex}");
                            
                            // اگر آیتم پیدا نشد
                            if ($currentIndex < 0) {
                                $error_message = "آیتم در لیست پیدا نشد";
                            } else {
                                $targetItem = null;
                                
                                // بررسی و پیدا کردن target item
                                if ($direction === 'up') {
                                    // برای بالا بردن، باید با آیتم قبلی (index کمتر = display_order کمتر) جابجا شود
                                    if ($currentIndex > 0) {
                                        $targetItem = $sameLevelItems[$currentIndex - 1];
                                    } else {
                                        $error_message = "این آیتم در بالاترین موقعیت است (index=0)";
                                    }
                                } elseif ($direction === 'down') {
                                    // برای پایین بردن، باید با آیتم بعدی (index بیشتر = display_order بیشتر) جابجا شود
                                    if ($currentIndex < count($sameLevelItems) - 1) {
                                        $targetItem = $sameLevelItems[$currentIndex + 1];
                                    } else {
                                        $error_message = "این آیتم در پایین‌ترین موقعیت است (index=" . (count($sameLevelItems) - 1) . ")";
                                    }
                                }
                                
                                // اگر targetItem پیدا شد، جابجایی را انجام می‌دهیم
                                if ($targetItem && is_array($targetItem) && empty($error_message)) {
                                    // جابجایی display_order
                                    $targetOrder = isset($targetItem['display_order']) ? (int)$targetItem['display_order'] : 0;
                                    $targetId = (int)$targetItem['id'];
                                    
                                    // اگر همه display_order یکسان هستند، باید مقادیر جدید تخصیص دهیم
                                    if ($currentOrder == $targetOrder) {
                                        // تخصیص مقادیر جدید بر اساس جهت
                                        if ($direction === 'up') {
                                            // بالا بردن: آیتم فعلی باید بالاتر از target قرار بگیرد
                                            // پس display_order فعلی را کمتر از target می‌کنیم
                                            // اگر target 0 است، آیتم فعلی را -1 می‌کنیم که بعداً به 0 تبدیل می‌شود
                                            if ($targetOrder <= 0) {
                                                // اگر target 0 است، آیتم فعلی را 0 می‌گذاریم و target را 1
                                                $newCurrentOrder = 0;
                                                $newTargetOrder = 1;
                                            } else {
                                                $newCurrentOrder = $targetOrder - 1;
                                                $newTargetOrder = $targetOrder + 1;
                                            }
                                        } else {
                                            // پایین بردن: آیتم فعلی باید پایین‌تر از target قرار بگیرد
                                            // پس display_order فعلی را بیشتر از target می‌کنیم
                                            if ($targetOrder <= 0) {
                                                // اگر target 0 است، آیتم فعلی را 1 می‌گذاریم و target را 0
                                                $newCurrentOrder = 1;
                                                $newTargetOrder = 0;
                                            } else {
                                                $newCurrentOrder = $targetOrder + 1;
                                                $newTargetOrder = $targetOrder - 1;
                                            }
                                        }
                                    } else {
                                        // اگر display_order متفاوت است، فقط جابجا می‌کنیم
                                        $newCurrentOrder = $targetOrder;
                                        $newTargetOrder = $currentOrder;
                                    }
                                    
                                    // لاگ برای دیباگ
                                    error_log("Move order: swapping id={$id} (order {$currentOrder} -> {$newCurrentOrder}) with id={$targetId} (order {$targetOrder} -> {$newTargetOrder})");
                                    
                                    // انجام update
                                    $result1 = $models[$section]->update($id, ['display_order' => $newCurrentOrder]);
                                    $result2 = $models[$section]->update($targetId, ['display_order' => $newTargetOrder]);
                                    
                                    if ($result1 && $result2) {
                                        // موفقیت - redirect
                                        $redirectUrl = "?section={$section}";
                                        if ($parentIdParam !== null && $parentIdParam > 0) {
                                            $redirectUrl .= "&parent_id={$parentIdParam}";
                                        } elseif ($currentParentId !== null && $currentParentId > 0) {
                                            $redirectUrl .= "&parent_id={$currentParentId}";
                                        }
                                        header("Location: " . $redirectUrl);
                                        exit;
                                    } else {
                                        // لاگ خطا برای دیباگ
                                        error_log("Move order failed: result1=" . ($result1 ? 'true' : 'false') . ", result2=" . ($result2 ? 'true' : 'false') . ", section={$section}, id={$id}, targetId={$targetId}");
                                        $error_message = "خطا در تغییر ترتیب - لطفاً دوباره تلاش کنید";
                                    }
                                } else {
                                    // لاگ برای دیباگ
                                    $errorMsg = isset($error_message) ? $error_message : 'none';
                                    error_log("Move order: targetItem not found or error exists. currentIndex={$currentIndex}, direction={$direction}, sameLevelItems count=" . count($sameLevelItems) . ", error={$errorMsg}");
                                }
                            }
                        }
                    } else {
                        $error_message = "شما دسترسی لازم را ندارید یا آیتم پیدا نشد";
                    }
                } else {
                    $error_message = "اطلاعات ناقص است";
                }
            }
            break;
            
        case 'toggle_pin':
            // تغییر وضعیت پین
            $id = $_POST['id'] ?? 0;
            
            if (!empty($id) && !empty($section) && isset($models[$section])) {
                $item = $models[$section]->getById($id);
                if ($item) {
                    // بررسی دسترسی - اگر created_by null است یا کاربر خودش سازنده است یا super_admin است
                    $itemCreatorId = $item['created_by'] ?? null;
                    $canModify = false;
                    
                    if ($itemCreatorId === null || $itemCreatorId === '') {
                        // اگر created_by null است، همه می‌توانند پین کنند
                        $canModify = true;
                    } else {
                        // بررسی دسترسی با canModifyContent
                        $canModify = $adminModel->canModifyContent($currentUser['id'], $itemCreatorId, $currentUser['role']);
                    }
                    
                    if ($canModify) {
                        // بررسی وضعیت فعلی پین - استفاده از isset و بررسی مقدار
                        $currentPinStatus = isset($item['is_pinned']) && $item['is_pinned'] !== null ? (int)$item['is_pinned'] : 0;
                        $newPinStatus = ($currentPinStatus === 1) ? 0 : 1;
                        
                        error_log("Toggle pin - Section: {$section}, ID: {$id}, Current: {$currentPinStatus}, New: {$newPinStatus}");
                        
                        // بررسی وجود ستون is_pinned در جدول و اضافه کردن در صورت عدم وجود
                        if (!$models[$section]->columnExists('is_pinned')) {
                            error_log("Column is_pinned does not exist in table for section: {$section}, attempting to add it");
                            // تلاش برای اضافه کردن ستون
                            if (!$models[$section]->addIsPinnedColumnIfNotExists()) {
                                error_log("Failed to add is_pinned column to table for section: {$section}");
                                $error_message = "خطا در اضافه کردن ستون پین. لطفاً با مدیر سیستم تماس بگیرید.";
                                $result = false;
                                break;
                            }
                            error_log("Successfully added is_pinned column to table for section: {$section}");
                            // بعد از اضافه کردن ستون، دوباره آیتم را دریافت می‌کنیم
                            $item = $models[$section]->getById($id);
                            $currentPinStatus = isset($item['is_pinned']) && $item['is_pinned'] !== null ? (int)$item['is_pinned'] : 0;
                            $newPinStatus = ($currentPinStatus === 1) ? 0 : 1;
                        }
                        
                        // به‌روزرسانی وضعیت پین
                        $result = $models[$section]->update($id, ['is_pinned' => $newPinStatus]);
                        if ($result) {
                            // Redirect با حفظ parent_id
                            $redirectUrl = "?section={$section}";
                            $parentIdParam = $_POST['parent_id'] ?? null;
                            if ($parentIdParam !== null && $parentIdParam !== '') {
                                $redirectUrl .= "&parent_id=" . (int)$parentIdParam;
                            }
                            header("Location: " . $redirectUrl);
                            exit;
                        } else {
                            error_log("Toggle pin failed - Section: {$section}, ID: {$id}");
                            $error_message = "خطا در تغییر وضعیت پین";
                        }
                    } else {
                        $error_message = "شما دسترسی لازم را ندارید";
                    }
                } else {
                    $error_message = "مطلب یافت نشد";
                }
            } else {
                $error_message = "اطلاعات ناقص است";
            }
            break;
    }
}

// دریافت آمار کلی
$stats = [];
foreach ($sectionNames as $sec => $name) {
    $stats[$sec] = $models[$sec]->countAll();
}

// اگر بخش خاصی انتخاب شده، مطالب آن را دریافت می‌کنیم
$items = [];
$totalPages = 1;
$currentPage = 1;
$tagsByItem = [];

if (!empty($section) && isset($models[$section])) {
    $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    
    // دریافت تمام آیتم‌ها (بدون محدودیت parent_id برای جستجو)
    // همه ادمین‌ها می‌توانند همه محتواها را ببینند
    $allItemsForSearch = $models[$section]->getAllWithCreator();
    $allItems = $allItemsForSearch;
    
    // فیلتر کردن بر اساس parent_id (فولدر/فایل)
    if (in_array($section, $fileBasedSections, true)) {
        if ($parentId === null) {
            // در سطح اصلی: فقط فولدرها و فایل‌هایی که parent_id = NULL
            $allItems = array_filter($allItems, function($item) {
                $itemParentId = $item['parent_id'] ?? null;
                return ($itemParentId === null || $itemParentId === '');
            });
        } else {
            // درون یک فولدر: فقط فولدرها و فایل‌هایی که parent_id = $parentId
            $allItems = array_filter($allItems, function($item) use ($parentId) {
                $itemParentId = isset($item['parent_id']) && $item['parent_id'] !== null && $item['parent_id'] !== '' 
                    ? (int)$item['parent_id'] 
                    : null;
                return $itemParentId !== null && $itemParentId === $parentId;
            });
        }
        $allItems = array_values($allItems);
        
        // مرتب‌سازی بر اساس display_order (فقط برای بخش‌هایی که display_order دارند)
        usort($allItems, function($a, $b) {
            // اول پین شده‌ها
            $pinnedA = !empty($a['is_pinned']) ? 1 : 0;
            $pinnedB = !empty($b['is_pinned']) ? 1 : 0;
            if ($pinnedA !== $pinnedB) {
                return $pinnedB - $pinnedA; // پین شده‌ها اول
            }
            
            $orderA = (int)($a['display_order'] ?? 0);
            $orderB = (int)($b['display_order'] ?? 0);
            if ($orderA === $orderB) {
                // اگر display_order یکسان است، بر اساس id مرتب می‌کنیم
                return (int)$a['id'] - (int)$b['id'];
            }
            return $orderA - $orderB;
        });
    } else {
        // برای بخش‌های متن ساده (مثل FAQ)، پین شده‌ها را در ابتدای لیست نمایش می‌دهیم
        usort($allItems, function($a, $b) {
            $pinnedA = !empty($a['is_pinned']) ? 1 : 0;
            $pinnedB = !empty($b['is_pinned']) ? 1 : 0;
            if ($pinnedA !== $pinnedB) {
                return $pinnedB - $pinnedA;
            }
            $dateA = strtotime($a['created_at'] ?? '1970-01-01');
            $dateB = strtotime($b['created_at'] ?? '1970-01-01');
            return $dateB <=> $dateA;
        });
    }
    
    // اعمال فیلترهای جستجو (قبل از فیلتر parent_id)
    $searchTerm = isset($_GET['search']) ? trim($_GET['search']) : '';
    $filterCategory = isset($_GET['filter_category']) ? (int)$_GET['filter_category'] : 0;
    $filterCreator = isset($_GET['filter_creator']) ? (int)$_GET['filter_creator'] : 0;
    $filterPinned = isset($_GET['filter_pinned']) ? $_GET['filter_pinned'] : '';
    
    // اگر جستجو یا فیلتر وجود دارد، از همه آیتم‌ها استفاده می‌کنیم (نه فقط سطح فعلی)
    if (!empty($searchTerm) || $filterCategory > 0 || $filterCreator > 0 || $filterPinned !== '') {
        $allItems = array_filter($allItemsForSearch, function($item) use ($searchTerm, $filterCategory, $filterCreator, $filterPinned, $section) {
            // فیلتر جستجو - جستجو در عنوان، توضیحات، بدنه و تگ‌ها
            if (!empty($searchTerm)) {
                $title = $section === 'faq' ? ($item['question'] ?? '') : ($item['title'] ?? '');
                $description = $item['description'] ?? '';
                $body = $item['body'] ?? '';
                
                $found = false;
                // جستجو در عنوان
                if (stripos($title, $searchTerm) !== false) {
                    $found = true;
                }
                // جستجو در توضیحات
                if (!$found && !empty($description) && stripos($description, $searchTerm) !== false) {
                    $found = true;
                }
                // جستجو در بدنه خبر
                if (!$found && !empty($body) && stripos($body, $searchTerm) !== false) {
                    $found = true;
                }
                // جستجو در تگ‌ها
                if (!$found) {
                    $itemTags = $tagModel->getTagsForItems($section, [$item['id']]);
                    if (!empty($itemTags[$item['id']] ?? [])) {
                        foreach ($itemTags[$item['id']] as $tag) {
                            if (stripos($tag, $searchTerm) !== false) {
                                $found = true;
                                break;
                            }
                        }
                    }
                }
                
                if (!$found) {
                    return false;
                }
            }
            
            // فیلتر دسته‌بندی
            if ($filterCategory > 0) {
                $itemParentId = isset($item['parent_id']) && $item['parent_id'] !== null && $item['parent_id'] !== '' 
                    ? (int)$item['parent_id'] 
                    : null;
                if ($itemParentId !== $filterCategory) {
                    return false;
                }
            }
            
            // فیلتر سازنده
            if ($filterCreator > 0) {
                $itemCreatorId = isset($item['created_by']) ? (int)$item['created_by'] : 0;
                if ($itemCreatorId !== $filterCreator) {
                    return false;
                }
            }
            
            // فیلتر پین شده
            if ($filterPinned !== '') {
                $isPinned = !empty($item['is_pinned']) ? 1 : 0;
                $filterPinnedValue = (int)$filterPinned;
                if ($isPinned !== $filterPinnedValue) {
                    return false;
                }
            }
            
            return true;
        });
        $allItems = array_values($allItems);
    }
    
    // نمایش همه مطالب بدون pagination
    $totalItems = count($allItems);
    $totalPages = 1;
    $items = $allItems; // نمایش همه مطالب

    if (!empty($items)) {
        $itemIds = array_column($items, 'id');
        $tagsByItem = $tagModel->getTagsForItems($section, $itemIds);
    }
}

// دریافت لیست ادمین‌ها (فقط برای سوپر ادمین)
$allAdmins = [];
if (canManageAdmins()) {
    $allAdmins = $adminModel->getAllAdmins();
    // Debug: بررسی تعداد ادمین‌ها
    error_log("تعداد ادمین‌ها: " . count($allAdmins));
}

?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>پنل مدیریت - QSCE</title>
    <link rel="icon" type="image/png" href="../image/bot.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/gh/rastikerdar/vazir-font@v30.1.0/dist/font-face.css" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/persian-datepicker@1.2.0/dist/css/persian-datepicker.min.css">
    <link rel="stylesheet" href="https://unpkg.com/@babakhani/persian-datepicker@latest/dist/css/persian-datepicker.min.css">
    <script>
        // تعریف تابع در scope global برای دسترسی از onchange در HTML
        window.toggleContentType = function(mode) {
            console.log('toggleContentType called with mode:', mode);
            
            var selectElement = document.getElementById('content_type_' + mode);
            if (!selectElement) {
                console.error('Select element not found: content_type_' + mode);
                setTimeout(function() {
                    window.toggleContentType(mode);
                }, 100);
                return;
            }
            
            var contentType = selectElement.value;
            var telegramLink = document.getElementById('telegram_link_' + mode);
            var fileUpload = document.getElementById('file_upload_' + mode);
            var telegramInput = document.getElementById('telegram_message_link_' + mode);
            var fileInput = document.getElementById('file_input_' + mode);
            
            if (contentType === 'link') {
                if (telegramLink) {
                    telegramLink.style.setProperty('display', 'block', 'important');
                    telegramLink.style.setProperty('visibility', 'visible', 'important');
                }
                if (telegramInput) {
                    telegramInput.setAttribute('required', 'required');
                }
                if (fileUpload) {
                    fileUpload.style.setProperty('display', 'none', 'important');
                    fileUpload.style.setProperty('visibility', 'hidden', 'important');
                }
                if (fileInput) {
                    fileInput.removeAttribute('required');
                }
            } else {
                if (telegramLink) {
                    telegramLink.style.setProperty('display', 'none', 'important');
                    telegramLink.style.setProperty('visibility', 'hidden', 'important');
                }
                if (telegramInput) {
                    telegramInput.removeAttribute('required');
                }
                if (fileUpload) {
                    fileUpload.style.setProperty('display', 'block', 'important');
                    fileUpload.style.setProperty('visibility', 'visible', 'important');
                }
                if (fileInput) {
                    fileInput.setAttribute('required', 'required');
                }
            }
        };
    </script>
    <style>
        /* استایل اضافی برای تقویم شمسی */
        .pdp-container {
            z-index: 9999 !important;
        }
        .pwt-datepicker {
            z-index: 9999 !important;
        }
    </style>
    <style>
        body {
            background: #f5f7fb;
            font-family: 'Vazir', 'Tahoma', sans-serif;
            color: #111827;
        }
        a {
            text-decoration: none;
        }
        .sidebar {
            background: #ffffff;
            min-height: 100vh;
            color: #1f2937;
            border-left: 1px solid #e5e7eb;
            box-shadow: 4px 0 24px rgba(15, 23, 42, 0.04);
        }
        .sidebar-header {
            color: #1f2937;
        }
        .sidebar-header i {
            color: #4f46e5;
        }
        .sidebar-header .role-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.35rem;
        }
        .sidebar .nav {
            padding: 0 1.25rem 2rem;
        }
        .sidebar .nav-link {
            color: #4b5563;
            padding: 12px 18px;
            border-left: 3px solid transparent;
            border-radius: 12px;
            transition: all 0.25s ease;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 0.35rem;
        }
        .sidebar .nav-link i {
            font-size: 1rem;
            margin-left: 0;
            width: auto;
        }
        .sidebar .nav-link.btn-danger {
            background: rgba(220, 53, 69, 0.08);
            color: #dc3545;
            border-radius: 12px;
        }
        .sidebar .nav-link.btn-danger:hover {
            background: rgba(220, 53, 69, 0.15);
            color: #b02a37;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(99, 102, 241, 0.1);
            color: #4338ca;
            border-left-color: #4f46e5;
            box-shadow: inset 0 0 0 1px rgba(79, 70, 229, 0.08);
        }
        .sidebar hr {
            opacity: 1;
            border-top: 1px solid #e5e7eb;
        }
        .content-area {
            padding: 30px;
        }
        .stat-card {
            background: #ffffff;
            border-radius: 18px;
            padding: 24px;
            box-shadow: 0 20px 45px rgba(15, 23, 42, 0.08);
            margin-bottom: 20px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .stat-card:hover {
            transform: translateY(-6px);
            box-shadow: 0 24px 60px rgba(79, 70, 229, 0.12);
        }
        .stat-icon {
            font-size: 3rem;
            color: #4f46e5;
        }
        .upload-progress .progress {
            height: 1.25rem;
        }
        .upload-progress .progress-bar {
            font-weight: bold;
        }
        /* استایل دکمه‌های عملیات کوچک‌تر */
        .operations-buttons {
            display: flex;
            align-items: center;
            gap: 4px;
            flex-wrap: wrap;
        }
        .btn-xs {
            padding: 0.2rem 0.4rem;
            font-size: 0.75rem;
            line-height: 1.2;
            border-radius: 4px;
        }
        .btn-xs i {
            font-size: 0.7rem;
        }
        .role-badge {
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
        }
        .role-admin {
            background: #17a2b8;
            color: white;
        }
        .role-super-admin {
            background: #dc3545;
            color: white;
        }
        .jalali-datetime {
            display: inline-flex;
            align-items: center;
            gap: 0.6rem;
            direction: rtl;
        }
        .jalali-datetime .jalali-date {
            font-weight: 400;
            color: #111827;
        }
        .jalali-datetime .jalali-time {
            font-size: 0.9rem;
            color: #6b7280;
        }
        /* استایل برای سطرهای جدول - hover و کلیک */
        .table tbody tr {
            transition: all 0.2s ease;
            cursor: pointer;
        }
        .table tbody tr:hover {
            background-color: #f0f4ff !important;
            transform: translateX(-3px);
            box-shadow: 0 2px 8px rgba(79, 70, 229, 0.15);
        }
        /* مخفی کردن scrollbar افقی در table-responsive */
        .table-responsive {
            overflow-x: auto;
            overflow-y: visible;
        }
        .table-responsive::-webkit-scrollbar {
            display: none;
        }
        .table-responsive {
            -ms-overflow-style: none;
            scrollbar-width: none;
        }
        .table tbody tr:active {
            transform: translateX(-1px);
        }
        /* استایل برای مطالب پین شده */
        .pinned-item {
            background: linear-gradient(90deg, #fff3cd 0%, #ffffff 5%) !important;
            border-right: 4px solid #ffc107 !important;
        }
        .pinned-item:hover {
            background: linear-gradient(90deg, #ffe69c 0%, #ffffff 5%) !important;
        }
        .pinned-item td {
            position: relative;
        }
        /* Badge پین در سلول عنوان */
        .pinned-badge-inline {
            display: inline-block;
            margin-left: 8px;
            font-size: 0.9rem;
            opacity: 0.7;
            vertical-align: middle;
        }
        .pinned-badge {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 5px;
            background: #ffc107;
            color: #000;
            padding: 5px 12px;
            border-radius: 15px;
            font-size: 0.85rem;
            font-weight: 600;
            white-space: nowrap;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .pinned-badge i {
            font-size: 0.9rem;
        }
        /* استایل مینیمال و مدرن برای تگ‌های محتوا - بنفش کمرنگ */
        .content-tag-badge {
            display: inline-flex;
            align-items: center;
            padding: 5px 12px;
            background: linear-gradient(135deg, #f5f3ff 0%, #ede9fe 100%);
            color: #6d28d9;
            border: 1.5px solid #c4b5fd;
            border-radius: 12px;
            font-size: 0.73rem;
            font-weight: 500;
            white-space: nowrap;
            transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1);
            margin: 3px 2px;
            box-shadow: 0 2px 4px rgba(109, 40, 217, 0.08), 0 1px 2px rgba(109, 40, 217, 0.04);
            position: relative;
            overflow: hidden;
        }
        .content-tag-badge::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
            transition: left 0.5s;
        }
        .content-tag-badge:hover {
            background: linear-gradient(135deg, #ede9fe 0%, #ddd6fe 100%);
            border-color: #a78bfa;
            color: #5b21b6;
            box-shadow: 0 4px 8px rgba(109, 40, 217, 0.15), 0 2px 4px rgba(109, 40, 217, 0.1);
            transform: translateY(-2px) scale(1.02);
        }
        .content-tag-badge:hover::before {
            left: 100%;
        }
        /* استایل نوتیفیکیشن داینامیک */
        #notificationContainer {
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 10000;
            pointer-events: none;
            max-width: 500px;
            width: 90%;
        }
        .dynamic-notification {
            background: #ffffff;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            padding: 16px 20px;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            gap: 12px;
            min-width: 300px;
            max-width: 500px;
            animation: slideDown 0.4s ease-out;
            border-right: 4px solid;
            position: relative;
            overflow: hidden;
            pointer-events: auto;
        }
        .dynamic-notification.success {
            border-color: #10b981;
            background: linear-gradient(90deg, #ecfdf5 0%, #ffffff 5%);
        }
        .dynamic-notification.error {
            border-color: #ef4444;
            background: linear-gradient(90deg, #fef2f2 0%, #ffffff 5%);
        }
        .dynamic-notification.warning {
            border-color: #f59e0b;
            background: linear-gradient(90deg, #fffbeb 0%, #ffffff 5%);
        }
        .dynamic-notification.info {
            border-color: #3b82f6;
            background: linear-gradient(90deg, #eff6ff 0%, #ffffff 5%);
        }
        .dynamic-notification .notification-icon {
            font-size: 1.5rem;
            flex-shrink: 0;
        }
        .dynamic-notification.success .notification-icon {
            color: #10b981;
        }
        .dynamic-notification.error .notification-icon {
            color: #ef4444;
        }
        .dynamic-notification.warning .notification-icon {
            color: #f59e0b;
        }
        .dynamic-notification.info .notification-icon {
            color: #3b82f6;
        }
        .dynamic-notification .notification-content {
            flex-grow: 1;
            font-weight: 500;
            color: #1f2937;
            font-size: 0.95rem;
        }
        .dynamic-notification .notification-close {
            background: none;
            border: none;
            color: #6b7280;
            font-size: 1.2rem;
            cursor: pointer;
            padding: 0;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.2s;
            flex-shrink: 0;
        }
        .dynamic-notification .notification-close:hover {
            background: rgba(0, 0, 0, 0.05);
            color: #1f2937;
        }
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        @keyframes slideUp {
            from {
                opacity: 1;
                transform: translateY(0);
                max-height: 100px;
                margin-bottom: 12px;
            }
            to {
                opacity: 0;
                transform: translateY(-10px);
                max-height: 0;
                margin-bottom: 0;
                padding: 0;
            }
        }
        /* منوی راست کلیک */
        .context-menu {
            position: fixed;
            background: white;
            border: 1px solid #ddd;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            z-index: 10000;
            min-width: 200px;
            display: none;
            padding: 4px 0;
        }
        .context-menu-item {
            padding: 10px 20px;
            cursor: pointer;
            transition: background 0.2s;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .context-menu-item:hover {
            background: #f0f4ff;
        }
        .context-menu-item.danger {
            color: #dc3545;
        }
        .context-menu-item.danger:hover {
            background: #fee;
        }
        .context-menu-separator {
            height: 1px;
            background: #e5e7eb;
            margin: 4px 0;
        }
        .table-row-hover:hover {
            background-color: #f8f9fa !important;
        }
        /* بهبود استایل مودال */
        .modal-content {
            border-radius: 20px;
            border: none;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .modal-header {
            display: none; /* حذف هدر */
        }
        .modal-body {
            padding: 30px;
            background: #ffffff;
        }
        .modal-footer {
            border-top: 1px solid #e9ecef;
            padding: 20px 30px;
            background: #f8f9fa;
            border-radius: 0 0 20px 20px;
        }
        /* دکمه بستن در گوشه */
        .modal-close-btn {
            position: absolute;
            top: 15px;
            left: 15px;
            z-index: 1051;
            background: rgba(255,255,255,0.9);
            border: none;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s;
            box-shadow: 0 2px 8px rgba(0,0,0,0.15);
        }
        .modal-close-btn:hover {
            background: #fff;
            transform: rotate(90deg);
            box-shadow: 0 4px 12px rgba(0,0,0,0.2);
        }
        /* استایل دو ستونی برای فیلدهای کوتاه */
        .form-row-2 {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        @media (max-width: 768px) {
            .form-row-2 {
                grid-template-columns: 1fr;
            }
        }
        .form-field-full {
            grid-column: 1 / -1;
        }
        .form-label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 8px;
        }
        /* استایل تقویم شمسی */
        .pdp-container,
        .pwt-datepicker,
        .pdp-container * {
            z-index: 10000 !important;
        }
        .pdp-container {
            position: fixed !important;
        }
        .form-control, .form-select {
            border-radius: 8px;
            border: 1px solid #ced4da;
            padding: 10px 15px;
            transition: all 0.3s;
        }
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
            transform: translateY(-1px);
        }
        /* فلش select box در سمت چپ */
        .form-select {
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3e%3cpath fill='none' stroke='%23343a40' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M2 5l6 6 6-6'/%3e%3c/svg%3e");
            background-position: left 0.75rem center;
            background-repeat: no-repeat;
            background-size: 16px 12px;
            padding-left: 2.5rem;
            padding-right: 0.75rem;
        }
        .form-select[dir="rtl"] {
            background-position: right 0.75rem center;
            padding-right: 2.5rem;
            padding-left: 0.75rem;
        }
        .btn {
            border-radius: 8px;
            padding: 10px 20px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .input-group-text {
            border-radius: 8px 0 0 8px;
        }
        /* بهبود مودال برای دسکتاپ */
        @media (min-width: 992px) {
            .modal-dialog {
                max-width: 800px;
            }
        }
        @media (min-width: 1200px) {
            .modal-dialog {
                max-width: 900px;
            }
        }
        /* بهبود فرم‌ها */
        .form-control:focus, .form-select:focus {
            border-color: #4f46e5;
            box-shadow: 0 0 0 0.2rem rgba(79, 70, 229, 0.25);
        }
        .input-group .btn {
            border-color: #ced4da;
        }
        .input-group .btn:hover {
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar px-0">
                <div class="text-center py-4 sidebar-header">
                    <img src="../image/bot.png" alt="Bot" style="width: 80px; height: 80px; margin-bottom: 1rem; border-radius: 50%; border: 3px solid #4f46e5; box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);">
                    <h4 style="font-size: 0.95rem; font-weight: 600;">بات جامعه مهندسین ساختمان استان قم</h4>
                    <h5 style="font-size: 1.1rem; font-weight: bold; color: #4f46e5;">QSCE</h5>
                    <p class="small mb-2" style="font-size: 0.85rem; color: #6b7280;">توسعه‌دهنده: سعید خلیلی</p>
                    <p class="small mb-2" style="font-weight: 600;"><?php echo htmlspecialchars($currentUser['full_name']); ?></p>
                    <span class="role-badge <?php echo $currentUser['role'] === 'super_admin' ? 'role-super-admin' : 'role-admin'; ?>">
                        <?php echo $currentUser['role'] === 'super_admin' ? '🔴 سوپر ادمین' : '🔵 ادمین عادی'; ?>
                    </span>
                    </div>
                    <nav class="nav flex-column">
                    <a class="nav-link <?php echo empty($section) ? 'active' : ''; ?>" href="index.php">
                        <i class="fas fa-home"></i> داشبورد
                    </a>
                    <?php foreach ($sectionNames as $sec => $name): ?>
                        <a class="nav-link <?php echo $section === $sec ? 'active' : ''; ?>" href="index.php?section=<?php echo $sec; ?>">
                            <i class="fas fa-folder"></i> <?php echo $name; ?>
                        </a>
                    <?php endforeach; ?>
                    <?php if ($currentUser['role'] === 'super_admin'): ?>
                        <hr class="my-3">
                        <a class="nav-link <?php echo $section === 'broadcast' ? 'active' : ''; ?>" href="index.php?section=broadcast">
                            <i class="fas fa-bullhorn"></i> ارسال پیام همگانی
                        </a>
                    <?php endif; ?>
                    <?php if (canManageAdmins()): ?>
                        <hr class="my-3">
                        <a class="nav-link" href="index.php?section=admins">
                            <i class="fas fa-users-cog"></i> مدیریت ادمین‌ها
                        </a>
                    <?php endif; ?>
                    <hr class="my-3">
                    <a class="nav-link <?php echo $section === 'users' ? 'active' : ''; ?>" href="index.php?section=users">
                        <i class="fas fa-users"></i> کاربران
                    </a>
                    <hr class="my-3">
                    <a class="nav-link" href="analytics.php">
                        <i class="fas fa-chart-bar"></i> آمار و آنالیز
                    </a>
                    <?php if ($currentUser['role'] === 'super_admin'): ?>
                        <hr class="my-3">
                        <a class="nav-link <?php echo $section === 'about_us' ? 'active' : ''; ?>" href="index.php?section=about_us">
                            <i class="fas fa-info-circle"></i> درباره ما
                        </a>
                    <?php endif; ?>
                    <hr class="my-3">
                    <a class="nav-link btn-danger <?php echo $section === 'change_password' ? 'active' : ''; ?>" href="index.php?section=change_password">
                        <i class="fas fa-key"></i> تغییر رمز عبور
                    </a>
                    <hr class="my-3">
                        <a class="nav-link" href="logout.php">
                            <i class="fas fa-sign-out-alt"></i> خروج
                        </a>
                    </nav>
            </div>
            
            <!-- Content Area -->
            <div class="col-md-10 content-area">
                <!-- Container برای نوتیفیکیشن‌های داینامیک -->
                <div id="notificationContainer" style="position: fixed; top: 20px; left: 50%; transform: translateX(-50%); z-index: 9999; width: 90%; max-width: 500px; pointer-events: none;"></div>
                        
                <?php if ($section === 'admins' && canManageAdmins()): ?>
                    <!-- بخش مدیریت ادمین‌ها -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>مدیریت ادمین‌ها</h2>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addAdminModal">
                            <i class="fas fa-user-plus me-2"></i>افزودن ادمین
                        </button>
                                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>نام کاربری</th>
                                    <th>نام کامل</th>
                                    <th>نقش</th>
                                    <th>تاریخ ثبت</th>
                                    <th>آخرین ورود</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($allAdmins as $admin): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($admin['username']); ?></td>
                                        <td><?php echo htmlspecialchars($admin['full_name']); ?></td>
                                        <td>
                                            <span class="role-badge <?php echo $admin['role'] === 'super_admin' ? 'role-super-admin' : 'role-admin'; ?>">
                                                <?php echo $admin['role'] === 'super_admin' ? 'سوپر ادمین' : 'ادمین عادی'; ?>
                                            </span>
                                        </td>
                                        <td><?php echo formatDateTimeHtml($admin['created_at']); ?></td>
                                        <td>
                                            <?php echo $admin['last_login'] ? formatDateTimeHtml($admin['last_login']) : 'هنوز وارد نشده'; ?>
                                        </td>
                                        <td>
                                            <?php if ($admin['id'] != $currentUser['id']): ?>
                                                <button class="btn btn-sm btn-primary me-1" data-bs-toggle="modal" data-bs-target="#editAdminModal<?php echo $admin['id']; ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button class="btn btn-sm btn-info me-1" onclick="openChangePasswordModal(<?php echo $admin['id']; ?>)">
                                                    <i class="fas fa-key"></i>
                                                </button>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('آیا مطمئن هستید که می‌خواهید این ادمین را حذف کنید؟')">
                                                    <input type="hidden" name="action" value="delete_admin">
                                                    <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger me-1">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <span class="text-muted"><i class="fas fa-user"></i> شما</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                                </div>
                    
                    <!-- Modal افزودن ادمین -->
                    <div class="modal fade" id="addAdminModal" tabindex="-1">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title">افزودن ادمین جدید</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                                <form method="POST">
                                    <div class="modal-body">
                                        <input type="hidden" name="action" value="add_admin">
                                        <div class="mb-3">
                                            <label class="form-label">نام کاربری</label>
                                            <input type="text" name="username" class="form-control" required>
                                    </div>
                                        <div class="mb-3">
                                            <label class="form-label">رمز عبور</label>
                                            <input type="password" name="password" class="form-control" required>
                                </div>
                                        <div class="mb-3">
                                            <label class="form-label">نام کامل</label>
                                            <input type="text" name="full_name" class="form-control" required>
                            </div>
                                        <div class="mb-3">
                                            <label class="form-label">نقش</label>
                                            <select name="role" class="form-select">
                                                <option value="admin">ادمین عادی</option>
                                                <option value="super_admin">سوپر ادمین</option>
                                            </select>
                                    </div>
                                </div>
                                    <div class="modal-footer">
                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">لغو</button>
                                        <button type="submit" class="btn btn-primary">افزودن</button>
                            </div>
                                </form>
                                    </div>
                                </div>
                            </div>
                    
                    <!-- Modal های ویرایش ادمین -->
                    <?php foreach ($allAdmins as $admin): ?>
                        <?php if ($admin['id'] != $currentUser['id']): ?>
                            <div class="modal fade" id="editAdminModal<?php echo $admin['id']; ?>" tabindex="-1">
                                <div class="modal-dialog">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title">ویرایش ادمین</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                    </div>
                                        <form method="POST">
                                            <div class="modal-body">
                                                <input type="hidden" name="action" value="edit_admin">
                                                <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">نام کاربری</label>
                                                    <input type="text" name="username" class="form-control" value="<?php echo htmlspecialchars($admin['username']); ?>" required>
                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">نام کامل</label>
                                                    <input type="text" name="full_name" class="form-control" value="<?php echo htmlspecialchars($admin['full_name']); ?>" readonly>
                                                    <div class="form-text">نام کامل قابل تغییر نیست</div>
                            </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">نقش</label>
                                                    <select name="role" class="form-select">
                                                        <option value="admin" <?php echo $admin['role'] === 'admin' ? 'selected' : ''; ?>>ادمین عادی</option>
                                                        <option value="super_admin" <?php echo $admin['role'] === 'super_admin' ? 'selected' : ''; ?>>سوپر ادمین</option>
                                                    </select>
                                    </div>
                                </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">لغو</button>
                                                <button type="submit" class="btn btn-primary">ذخیره تغییرات</button>
                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                    
<?php elseif ($section === 'users'): ?>
                    <!-- بخش مدیریت کاربران بات -->
                    <?php
                    // دریافت کاربران
                    $usersPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;
                    $usersLimit = 50;
                    $usersOffset = ($usersPage - 1) * $usersLimit;
                    $totalUsers = $botUserModel->getTotalUsers();
                    $totalUsersPages = ceil($totalUsers / $usersLimit);
                    $users = $botUserModel->getAllUsers($usersLimit, $usersOffset, 'activity_count DESC, last_seen_at DESC');
                    ?>
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>کاربران بات</h2>
                        <div class="text-muted">
                            <i class="fas fa-users me-2"></i>
                            مجموع کاربران: <?php echo number_format($totalUsers); ?>
                        </div>
                    </div>
                    
                    <div class="card">
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>نام کاربری</th>
                                            <th>نام</th>
                                            <th>تعداد فعالیت</th>
                                            <th>اولین استفاده</th>
                                            <th>آخرین استفاده</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($users)): ?>
                                            <tr>
                                                <td colspan="6" class="text-center text-muted">
                                                    <i class="fas fa-inbox me-2"></i>
                                                    هنوز کاربری ثبت نشده است
                                                </td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($users as $index => $user): ?>
                                                <tr>
                                                    <td><?php echo $usersOffset + $index + 1; ?></td>
                                                    <td>
                                                        <?php if (!empty($user['username'])): ?>
                                                            <i class="fas fa-at me-1 text-muted"></i>
                                                            <?php echo htmlspecialchars($user['username']); ?>
                                                        <?php else: ?>
                                                            <span class="text-muted">—</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <?php 
                                                        $fullName = trim(($user['first_name'] ?? '') . ' ' . ($user['last_name'] ?? ''));
                                                        echo !empty($fullName) ? htmlspecialchars($fullName) : '<span class="text-muted">نامشخص</span>';
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <span class="badge bg-primary">
                                                            <?php echo number_format($user['activity_count'] ?? 0); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <?php 
                                                        if (!empty($user['first_seen_at'])) {
                                                            echo formatDateTimeHtml($user['first_seen_at']);
                                                        } else {
                                                            echo '<span class="text-muted">—</span>';
                                                        }
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <?php 
                                                        if (!empty($user['last_seen_at'])) {
                                                            echo formatDateTimeHtml($user['last_seen_at']);
                                                        } else {
                                                            echo '<span class="text-muted">—</span>';
                                                        }
                                                        ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalUsersPages > 1): ?>
                                <nav aria-label="Page navigation">
                                    <ul class="pagination justify-content-center mt-4">
                                        <?php if ($usersPage > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?section=users&page=<?php echo $usersPage - 1; ?>">
                                                    <i class="fas fa-chevron-right"></i> قبلی
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                        
                                        <?php
                                        $startPage = max(1, $usersPage - 2);
                                        $endPage = min($totalUsersPages, $usersPage + 2);
                                        
                                        for ($i = $startPage; $i <= $endPage; $i++):
                                        ?>
                                            <li class="page-item <?php echo $i === $usersPage ? 'active' : ''; ?>">
                                                <a class="page-link" href="?section=users&page=<?php echo $i; ?>">
                                                    <?php echo $i; ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>
                                        
                                        <?php if ($usersPage < $totalUsersPages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?section=users&page=<?php echo $usersPage + 1; ?>">
                                                    بعدی <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        </div>
                    </div>
                    
<?php elseif ($section === 'broadcast' && $currentUser['role'] === 'super_admin'): ?>
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>ارسال پیام همگانی</h2>
                    </div>
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">ارسال پیام جدید</h5>
                            <span class="text-muted small">ارسال با فاصله ۰٫۲ ثانیه برای هر کاربر</span>
                        </div>
                                    <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="create_broadcast">
                                <div class="mb-3">
                                    <label class="form-label">متن پیام</label>
                                    <textarea name="broadcast_message" class="form-control" rows="6" placeholder="متن پیام که برای همه کاربران ارسال می‌شود..." required></textarea>
                                    <div class="form-text">می‌توانید از تگ‌های HTML ساده برای قالب‌بندی استفاده کنید.</div>
                                    </div>
                                <div class="mb-3 d-flex justify-content-between align-items-center">
                                    <div>
                                        <label class="form-label me-3">فرمت پیام</label>
                                        <div class="form-check form-check-inline">
                                            <input class="form-check-input" type="radio" name="parse_mode" id="parse_mode_html" value="HTML" checked>
                                            <label class="form-check-label" for="parse_mode_html">HTML</label>
                                </div>
                                        <div class="form-check form-check-inline">
                                            <input class="form-check-input" type="radio" name="parse_mode" id="parse_mode_none" value="None">
                                            <label class="form-check-label" for="parse_mode_none">متن ساده</label>
                            </div>
                                    </div>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-paper-plane me-2"></i>ارسال برای همه کاربران
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    <?php 
                    $broadcasts = $broadcastModel->getAllMessages(100);
                    // دریافت اطلاعات سازنده برای هر broadcast
                    $broadcastCreators = [];
                    foreach ($broadcasts as $broadcast) {
                        if (!empty($broadcast['created_by'])) {
                            $creator = $adminModel->getById($broadcast['created_by']);
                            if ($creator) {
                                $broadcastCreators[$broadcast['id']] = $creator['full_name'] ?? $creator['username'] ?? 'نامشخص';
                            }
                        }
                    }
                    ?>
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">تاریخچه پیام‌های همگانی</h5>
                        </div>
                                    <div class="card-body">
                            <?php if (empty($broadcasts)): ?>
                                <div class="alert alert-info mb-0">
                                    هنوز پیامی ارسال نشده است.
                                    </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-striped align-middle">
                                        <thead>
                                            <tr>
                                                <th>شناسه</th>
                                                <th>متن پیام</th>
                                                <th>وضعیت</th>
                                                <th>کل گیرندگان</th>
                                                <th>موفق</th>
                                                <th>ناموفق</th>
                                                <th>ارسال شده توسط</th>
                                                <th>ایجاد</th>
                                                <th>شروع ارسال</th>
                                                <th>پایان</th>
                                                <th>آخرین خطا</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $statusLabels = [
                                                'pending' => ['label' => 'در انتظار', 'class' => 'badge bg-warning text-dark'],
                                                'in_progress' => ['label' => 'در حال ارسال', 'class' => 'badge bg-info text-white'],
                                                'completed' => ['label' => 'کامل شده', 'class' => 'badge bg-success'],
                                                'failed' => ['label' => 'ناموفق', 'class' => 'badge bg-danger']
                                            ];
                                            foreach ($broadcasts as $broadcast): 
                                                $status = $broadcast['status'] ?? 'pending';
                                                $badge = $statusLabels[$status] ?? ['label' => $status, 'class' => 'badge bg-secondary'];
                                                $messagePreview = mb_substr($broadcast['message_text'], 0, 160);
                                                if (mb_strlen($broadcast['message_text']) > 160) {
                                                    $messagePreview .= '…';
                                                }
                                                $creatorName = $broadcastCreators[$broadcast['id']] ?? 'نامشخص';
                                            ?>
                                                <tr>
                                                    <td>#<?php echo (int)$broadcast['id']; ?></td>
                                                    <td class="small text-end" style="min-width: 220px; white-space: pre-wrap;">
                                                        <?php echo nl2br(htmlspecialchars($messagePreview)); ?>
                                                    </td>
                                                    <td class="text-center">
                                                        <span class="<?php echo $badge['class']; ?>" style="font-size: 0.85rem; padding: 0.45rem 0.75rem; border-radius: 20px;"><?php echo $badge['label']; ?></span>
                                                    </td>
                                                    <td><?php echo (int)($broadcast['total_recipients'] ?? 0); ?></td>
                                                    <td><?php echo (int)($broadcast['sent_success'] ?? 0); ?></td>
                                                    <td><?php echo (int)($broadcast['sent_failed'] ?? 0); ?></td>
                                                    <td>
                                                        <small class="text-muted">
                                                            <i class="fas fa-user me-1"></i>
                                                            <?php echo htmlspecialchars($creatorName); ?>
                                                        </small>
                                                    </td>
                                                    <td><?php echo formatDateTimeHtml($broadcast['created_at'] ?? ''); ?></td>
                                                    <td><?php echo $broadcast['started_at'] ? formatDateTimeHtml($broadcast['started_at']) : '—'; ?></td>
                                                    <td><?php echo $broadcast['completed_at'] ? formatDateTimeHtml($broadcast['completed_at']) : '—'; ?></td>
                                                    <td class="small text-danger"><?php echo htmlspecialchars($broadcast['last_error'] ?? ''); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                            </div>
                    </div>
                    
<?php elseif ($section === 'about_us' && $currentUser['role'] === 'super_admin'): ?>
                    <!-- بخش مدیریت درباره ما -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>درباره ما</h2>
                                    </div>
                    
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">ویرایش متن درباره ما</h5>
                                </div>
                                    <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="edit_about_us">
                                <div class="mb-3">
                                    <label class="form-label">متن درباره ما</label>
                                    <textarea name="about_text" class="form-control" rows="10" required><?php echo htmlspecialchars($settingsModel->getAboutUs() ?? ''); ?></textarea>
                                    <div class="form-text">این متن در بات تلگرام نمایش داده می‌شود</div>
                                    </div>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>ذخیره تغییرات
                                </button>
                            </form>
                                </div>
                            </div>
                    
                <?php elseif ($section === 'change_password'): ?>
                    <!-- بخش تغییر رمز عبور -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>تغییر رمز عبور</h2>
                    </div>
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">تغییر رمز عبور</h5>
                        </div>
                                    <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="change_my_password">
                                <div class="mb-3">
                                    <label class="form-label">رمز عبور فعلی</label>
                                    <input type="password" name="current_password" class="form-control" required>
                                    <div class="form-text">لطفاً رمز عبور فعلی خود را وارد کنید</div>
                                    </div>
                                <div class="mb-3">
                                    <label class="form-label">رمز عبور جدید</label>
                                    <input type="password" name="new_password" class="form-control" required minlength="6">
                                    <div class="form-text">رمز عبور جدید باید حداقل 6 کاراکتر باشد</div>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">تأیید رمز عبور جدید</label>
                                    <input type="password" name="confirm_password" class="form-control" required minlength="6">
                                    <div class="form-text">لطفاً رمز عبور جدید را دوباره وارد کنید</div>
                            </div>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>تغییر رمز عبور
                                </button>
                            </form>
                        </div>
                    </div>
                    
                <?php elseif (!empty($section)): ?>
                    <!-- بخش مدیریت مطالب -->
                            <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>
                            <?php echo $sectionNames[$section]; ?>
                            <?php if ($parentId !== null && in_array($section, $fileBasedSections, true) && $parentId > 0): ?>
                                <?php 
                                $parentItem = $models[$section]->getById($parentId);
                                if ($parentItem && is_array($parentItem)): 
                                ?>
                                    <span class="text-muted">/</span>
                                    <span class="text-muted"><?php echo htmlspecialchars($parentItem['title'] ?? ''); ?></span>
                                <?php endif; ?>
                            <?php endif; ?>
                        </h2>
                        <div>
                            <?php if ($parentId !== null && in_array($section, $fileBasedSections, true) && $parentId > 0): ?>
                                <?php 
                                $parentItem = $models[$section]->getById($parentId);
                                $backParentId = null;
                                if ($parentItem && is_array($parentItem) && !empty($parentItem['parent_id'])) {
                                    $backParentId = (int)$parentItem['parent_id'];
                                }
                                ?>
                                <a href="?section=<?php echo $section; ?><?php echo $backParentId !== null ? '&parent_id=' . $backParentId : ''; ?>" class="btn btn-secondary me-2">
                                    <i class="fas fa-arrow-right me-2"></i>بازگشت
                                </a>
                            <?php endif; ?>
                        </div>
                            </div>
                            
                                    <!-- جستجو و فیلتر -->
                                    <div class="card mb-3">
                                        <div class="card-body">
                                            <form method="GET" class="row g-3">
                                                <input type="hidden" name="section" value="<?php echo $section; ?>">
                                                <?php if ($parentId !== null): ?>
                                                <input type="hidden" name="parent_id" value="<?php echo $parentId; ?>">
                                                <?php endif; ?>
                                                <div class="col-md-3">
                                                    <label class="form-label">جستجو در عنوان</label>
                                                    <input type="text" name="search" class="form-control" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" placeholder="جستجو...">
                                                </div>
                                                <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                <div class="col-md-2">
                                                    <label class="form-label">دسته‌بندی</label>
                                                    <select name="filter_category" class="form-select">
                                                        <option value="">همه</option>
                                                        <?php 
                                                        $allCategories = array_filter($items, function($it) {
                                                            return ($it['item_type'] ?? 'file') === 'folder';
                                                        });
                                                        foreach ($allCategories as $cat): 
                                                        ?>
                                                        <option value="<?php echo $cat['id']; ?>" <?php echo (isset($_GET['filter_category']) && $_GET['filter_category'] == $cat['id']) ? 'selected' : ''; ?>>
                                                            <?php echo htmlspecialchars($cat['title']); ?>
                                                        </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </div>
                                                <?php endif; ?>
                                                <div class="col-md-2">
                                                    <label class="form-label">سازنده</label>
                                                    <select name="filter_creator" class="form-select">
                                                        <option value="">همه</option>
                                                        <?php 
                                                        $creators = [];
                                                        foreach ($items as $it) {
                                                            $creatorId = $it['created_by'] ?? null;
                                                            $creatorName = $it['creator_name'] ?? 'نامشخص';
                                                            if ($creatorId && !isset($creators[$creatorId])) {
                                                                $creators[$creatorId] = $creatorName;
                                                            }
                                                        }
                                                        foreach ($creators as $cId => $cName): 
                                                        ?>
                                                        <option value="<?php echo $cId; ?>" <?php echo (isset($_GET['filter_creator']) && $_GET['filter_creator'] == $cId) ? 'selected' : ''; ?>>
                                                            <?php echo htmlspecialchars($cName); ?>
                                                        </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </div>
                                                <div class="col-md-2">
                                                    <label class="form-label">پین شده</label>
                                                    <select name="filter_pinned" class="form-select">
                                                        <option value="">همه</option>
                                                        <option value="1" <?php echo (isset($_GET['filter_pinned']) && $_GET['filter_pinned'] == '1') ? 'selected' : ''; ?>>بله</option>
                                                        <option value="0" <?php echo (isset($_GET['filter_pinned']) && $_GET['filter_pinned'] == '0') ? 'selected' : ''; ?>>خیر</option>
                                                    </select>
                                                </div>
                                                <div class="col-md-3">
                                                    <label class="form-label">&nbsp;</label>
                                                    <button type="submit" class="btn btn-primary w-100">
                                                        <i class="fas fa-search me-2" style="margin-left: 0.5rem;"></i>جستجو
                                                    </button>
                                                </div>
                                            </form>
                                        </div>
                                    </div>
                                    
                                    <!-- دکمه‌های افزودن -->
                                    <div class="mb-3 d-flex gap-2">
                                        <?php if (in_array($section, $fileBasedSections, true)): ?>
                                        <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addFolderModal">
                                            <i class="fas fa-folder-plus me-2" style="margin-left: 0.5rem;"></i>افزودن فولدر
                                        </button>
                                        <?php endif; ?>
                                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addModal">
                                            <i class="fas fa-plus me-2" style="margin-left: 0.5rem;"></i>افزودن مطلب
                                        </button>
                                    </div>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-hover table-striped align-middle">
                                            <thead class="table-light">
                                                <tr>
                                                    <th style="width: 30%;">عنوان</th>
                                                    <th style="width: 15%;">ایجاد شده توسط</th>
                                                    <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                        <th style="width: 10%;">حجم فایل</th>
                                                    <?php endif; ?>
                                                    <th style="width: 15%;">تاریخ ایجاد</th>
                                                    <th style="width: 20%;" class="text-center">عملیات</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                <?php foreach ($items as $item): ?>
                                    <?php 
                                    $isPinned = !empty($item['is_pinned']);
                                    $itemType = in_array($section, $fileBasedSections, true) ? ($item['item_type'] ?? 'file') : 'file';
                                    $isFolder = $itemType === 'folder';
                                    $clickUrl = $isFolder ? "?section={$section}&parent_id={$item['id']}" : "#editModal{$item['id']}";
                                    $rowClass = $isPinned ? 'pinned-item' : '';
                                    ?>
                                    <tr class="<?php echo $rowClass; ?>" 
                                        data-item-id="<?php echo $item['id']; ?>"
                                        data-section="<?php echo $section; ?>"
                                        data-is-folder="<?php echo $isFolder ? '1' : '0'; ?>"
                                        data-is-pinned="<?php echo $isPinned ? '1' : '0'; ?>"
                                        onclick="<?php echo $isFolder ? "window.location.href='{$clickUrl}'" : "\$('#editModal{$item['id']}').modal('show')"; ?>"
                                        oncontextmenu="showContextMenu(event, <?php echo $item['id']; ?>, '<?php echo $section; ?>', <?php echo $isFolder ? 'true' : 'false'; ?>, <?php echo $isPinned ? 'true' : 'false'; ?>); return false;"
                                        style="cursor: pointer;">
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <?php if ($isPinned): ?>
                                                    <span class="pinned-badge-inline">📌</span>
                                                <?php endif; ?>
                                                <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                    <?php 
                                                    $itemType = $item['item_type'] ?? 'file';
                                                    if ($itemType === 'folder'): 
                                                    ?>
                                                        <i class="fas fa-folder text-warning me-2" style="margin-left: 0.5rem;"></i>
                                                        <div style="flex: 1;">
                                                            <a href="?section=<?php echo $section; ?>&parent_id=<?php echo $item['id']; ?>" 
                                                               style="text-decoration: none; color: inherit; font-weight: 600;" 
                                                               title="<?php echo htmlspecialchars($item['title']); ?>"
                                                               onclick="event.stopPropagation();">
                                                                <?php 
                                                                $title = htmlspecialchars($item['title']);
                                                                echo mb_strlen($title) > 50 ? mb_substr($title, 0, 50) . '...' : $title;
                                                                ?>
                                                            </a>
                                                            <?php if (!empty($tagsByItem[$item['id']] ?? [])): ?>
                                                                <div class="mt-2 d-flex flex-wrap gap-1" style="margin-top: 0.5rem !important;">
                                                                    <?php foreach ($tagsByItem[$item['id']] as $tag): ?>
                                                                        <span class="content-tag-badge">#<?php echo htmlspecialchars($tag); ?></span>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php else: ?>
                                                        <i class="fas fa-file text-primary me-2" style="margin-left: 0.5rem;"></i>
                                                        <div style="flex: 1;">
                                                            <span title="<?php echo htmlspecialchars($item['title']); ?>" style="font-weight: 500;">
                                                                <?php 
                                                                $title = htmlspecialchars($item['title']);
                                                                echo mb_strlen($title) > 50 ? mb_substr($title, 0, 50) . '...' : $title;
                                                                ?>
                                                            </span>
                                                            <?php if (!empty($tagsByItem[$item['id']] ?? [])): ?>
                                                                <div class="mt-2 d-flex flex-wrap gap-1" style="margin-top: 0.5rem !important;">
                                                                    <?php foreach ($tagsByItem[$item['id']] as $tag): ?>
                                                                        <span class="content-tag-badge">#<?php echo htmlspecialchars($tag); ?></span>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                <?php else: ?>
                                                    <?php
                                                    $displayText = '';
                                                    if ($section === 'faq') {
                                                        $displayText = $item['question'];
                                                    } elseif ($section === 'ethics') {
                                                        $displayText = $item['quote_text'];
                                                    } else {
                                                        $displayText = $item['title'];
                                                    }
                                                    $displayTextEscaped = htmlspecialchars($displayText);
                                                    ?>
                                                    <span title="<?php echo $displayTextEscaped; ?>" style="font-weight: 500;">
                                                        <?php echo mb_strlen($displayText) > 50 ? mb_substr($displayTextEscaped, 0, 50) . '...' : $displayTextEscaped; ?>
                                                    </span>
                                                    <?php if ($section !== 'ethics' && !empty($tagsByItem[$item['id']] ?? [])): ?>
                                                        <div class="mt-2 d-flex flex-wrap gap-1" style="margin-top: 0.5rem !important;">
                                                            <?php foreach ($tagsByItem[$item['id']] as $tag): ?>
                                                                <span class="content-tag-badge">#<?php echo htmlspecialchars($tag); ?></span>
                                                            <?php endforeach; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td><?php echo htmlspecialchars($item['creator_name'] ?? 'نامشخص'); ?></td>
                                        <?php if (in_array($section, $fileBasedSections, true)): ?>
                                            <td>
                                                <?php
                                                $sizeText = '—';
                                                if (($item['item_type'] ?? 'file') === 'file' && !empty($item['file_path'])) {
                                                    $absolutePath = __DIR__ . '/../' . $item['file_path'];
                                                    if (file_exists($absolutePath)) {
                                                        $sizeText = formatFileSize(filesize($absolutePath));
                                                    } else {
                                                        $sizeText = '<span class="text-danger">فایل یافت نشد</span>';
                                                    }
                                                }
                                                echo $sizeText;
                                                ?>
                                                            </td>
                                                        <?php endif; ?>
                                                        <td><?php echo formatDateTimeHtml($item['created_at']); ?></td>
                                                        <td onclick="event.stopPropagation();">
                                            <?php if ($adminModel->canModifyContent($currentUser['id'], $item['created_by'], $currentUser['role'])): ?>
                                                <div class="operations-buttons" role="group" onclick="event.stopPropagation();">
                                                <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                    <form method="POST" style="display: inline-block; margin: 0 3px;">
                                                        <input type="hidden" name="action" value="move_order">
                                                        <input type="hidden" name="section" value="<?php echo $section; ?>">
                                                        <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                                        <input type="hidden" name="direction" value="up">
                                                        <?php if ($parentId !== null): ?>
                                                        <input type="hidden" name="parent_id" value="<?php echo $parentId; ?>">
                                                        <?php endif; ?>
                                                        <button type="submit" class="btn btn-xs btn-outline-primary" title="بالا" onclick="event.stopPropagation();">
                                                            <i class="fas fa-arrow-up"></i>
                                                        </button>
                                                    </form>
                                                    <form method="POST" style="display: inline-block; margin: 0 3px;">
                                                        <input type="hidden" name="action" value="move_order">
                                                        <input type="hidden" name="section" value="<?php echo $section; ?>">
                                                        <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                                        <input type="hidden" name="direction" value="down">
                                                        <?php if ($parentId !== null): ?>
                                                        <input type="hidden" name="parent_id" value="<?php echo $parentId; ?>">
                                                        <?php endif; ?>
                                                        <button type="submit" class="btn btn-xs btn-outline-primary" title="پایین" onclick="event.stopPropagation();">
                                                            <i class="fas fa-arrow-down"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                    <button class="btn btn-xs btn-warning" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $item['id']; ?>" onclick="event.stopPropagation();" title="ویرایش" style="border-radius: 8px; margin: 0 3px;">
                                                                <i class="fas fa-edit"></i>
                                                            </button>
                                                    <?php if ($section !== 'ethics'): ?>
                                                    <form method="POST" style="display: inline-block; margin: 0 3px;" onsubmit="event.stopPropagation(); return true;" onclick="event.stopPropagation();">
                                                        <input type="hidden" name="action" value="toggle_pin">
                                                        <input type="hidden" name="section" value="<?php echo $section; ?>">
                                                        <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                                        <?php if ($parentId !== null): ?>
                                                        <input type="hidden" name="parent_id" value="<?php echo $parentId; ?>">
                                                        <?php endif; ?>
                                                        <button type="submit" class="btn btn-xs <?php echo !empty($item['is_pinned']) ? 'btn-success' : 'btn-outline-success'; ?>" title="<?php echo !empty($item['is_pinned']) ? 'برداشتن پین' : 'پین کردن'; ?>" onclick="event.stopPropagation(); return true;">
                                                            <i class="fas fa-thumbtack"></i>
                                                        </button>
                                                    </form>
                                                    <?php endif; ?>
                                                    <form method="POST" style="display: inline-block; margin: 0 3px;" onsubmit="return confirm('آیا مطمئن هستید؟')" onclick="event.stopPropagation();">
                                                        <input type="hidden" name="action" value="delete">
                                                        <input type="hidden" name="section" value="<?php echo $section; ?>">
                                                        <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                                        <button type="submit" class="btn btn-xs btn-danger" onclick="event.stopPropagation();">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-muted"><i class="fas fa-lock"></i> فقط نمایش</span>
                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                    
                                    <!-- منوی راست کلیک -->
                                    <div id="contextMenu" class="context-menu">
                                        <div class="context-menu-item" onclick="contextMenuAction('pin')">
                                            <i class="fas fa-thumbtack"></i>
                                            <span id="contextMenuPinText">پین کردن</span>
                                        </div>
                                        <div class="context-menu-item" onclick="contextMenuAction('move_up')">
                                            <i class="fas fa-arrow-up"></i>
                                            <span>انتقال به بالا</span>
                                        </div>
                                        <div class="context-menu-item" onclick="contextMenuAction('move_down')">
                                            <i class="fas fa-arrow-down"></i>
                                            <span>انتقال به پایین</span>
                                        </div>
                                        <div class="context-menu-separator"></div>
                                        <div class="context-menu-item" onclick="contextMenuAction('edit')">
                                            <i class="fas fa-edit"></i>
                                            <span>ویرایش</span>
                                        </div>
                                        <div class="context-menu-item danger" onclick="contextMenuAction('delete')">
                                            <i class="fas fa-trash"></i>
                                            <span>حذف</span>
                                        </div>
                                    </div>
                    
                    <!-- Modal های ویرایش -->
                    <?php foreach ($items as $item): ?>
                        <div class="modal fade" id="editModal<?php echo $item['id']; ?>" tabindex="-1">
                            <div class="modal-dialog modal-lg">
                                <div class="modal-content" style="position: relative;">
                                    <button type="button" class="modal-close-btn" data-bs-dismiss="modal" aria-label="بستن">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    <form method="POST" enctype="multipart/form-data" accept-charset="UTF-8" data-upload-form="true">
                                        <div class="modal-body">
                                            <input type="hidden" name="action" value="edit">
                                            <input type="hidden" name="section" value="<?php echo $section; ?>">
                                            <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                            
                                            <?php if ($section === 'faq'): ?>
                                                <div class="mb-3">
                                                    <label class="form-label">سؤال</label>
                                                    <input type="text" name="question" class="form-control" value="<?php echo htmlspecialchars($item['question']); ?>" required>
                                                </div>
                                                <div class="mb-3">
                                                    <label class="form-label">پاسخ</label>
                                                    <textarea name="answer" class="form-control" rows="4" required><?php echo htmlspecialchars($item['answer']); ?></textarea>
                                                </div>
                                            <?php elseif ($section === 'ai'): ?>
                                                <div class="mb-3">
                                                    <label class="form-label">عنوان</label>
                                                    <input type="text" name="title" class="form-control" value="<?php echo htmlspecialchars($item['title']); ?>" required>
                                                </div>
                                                <div class="mb-3">
                                                    <label class="form-label">لینک وبسایت</label>
                                                    <input type="url" name="website_url" class="form-control" value="<?php echo htmlspecialchars($item['website_url'] ?? ''); ?>" placeholder="https://example.com">
                                                </div>
                                            <?php elseif ($section === 'ethics'): ?>
                                                <div class="mb-3">
                                                    <label class="form-label">جمله</label>
                                                    <textarea name="quote_text" class="form-control" rows="4" required><?php echo htmlspecialchars($item['quote_text']); ?></textarea>
                                                </div>
                                            <?php else: ?>
                                                <div class="form-row-2">
                                                    <div class="mb-3 form-field-full">
                                                        <label class="form-label">عنوان</label>
                                                        <input type="text" name="title" class="form-control" value="<?php echo htmlspecialchars($item['title']); ?>" required>
                                                    </div>
                                                    <?php if ($section === 'news'): ?>
                                                    <div class="mb-3 form-field-full">
                                                        <label class="form-label">بدنه خبر</label>
                                                        <textarea name="body" class="form-control" rows="6" placeholder="متن کامل خبر را وارد کنید..."><?php echo htmlspecialchars($item['body'] ?? ''); ?></textarea>
                                                        <div class="form-text">متن کامل خبر که در بات نمایش داده می‌شود</div>
                                                    </div>
                                                    <div class="mb-3">
                                                        <label class="form-label">تاریخ خبر (شمسی)</label>
                                                        <div class="input-group">
                                                            <input type="text" name="persian_date" id="persian_date_edit<?php echo $item['id']; ?>" class="form-control" value="<?php echo htmlspecialchars($item['persian_date'] ?? ''); ?>" placeholder="1403/01/01">
                                                            <button type="button" class="btn btn-outline-info" onclick="setTodayPersianDate('edit<?php echo $item['id']; ?>')" title="تاریخ امروز">
                                                                <i class="fas fa-calendar-day"></i> امروز
                                                            </button>
                                                        </div>
                                                        <div class="form-text">تاریخ انتشار خبر به شمسی (مثال: 1403/01/01)</div>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                    <input type="hidden" name="item_type" value="<?php echo htmlspecialchars($item['item_type'] ?? 'file'); ?>">
                                                    <div class="mb-3 form-field-full" id="file_fields_edit<?php echo $item['id']; ?>" style="<?php echo (($item['item_type'] ?? 'file') === 'folder') ? 'display:none;' : ''; ?>">
                                                        <label class="form-label">نوع محتوا</label>
                                                        <select name="content_type" class="form-control" id="content_type_edit<?php echo $item['id']; ?>" onchange="toggleContentTypeEdit(<?php echo $item['id']; ?>)">
                                                            <option value="link" <?php echo empty($item['file_path']) ? 'selected' : ''; ?>>🔗  لینک</option>
                                                            <option value="file" <?php echo !empty($item['file_path']) ? 'selected' : ''; ?>>📄  آپلود فایل</option>
                                                        </select>
                                                        <div class="form-text">یکی از گزینه‌های بالا را انتخاب کنید</div>
                                                    </div>
                                                    <div class="mb-3 form-field-full" id="telegram_link_edit<?php echo $item['id']; ?>" style="<?php echo (($item['item_type'] ?? 'file') === 'folder' || !empty($item['file_path'])) ? 'display:none;' : ''; ?>">
                                                        <label class="form-label">لینک</label>
                                                        <input type="url" name="telegram_message_link" class="form-control" value="<?php echo htmlspecialchars(getItemLinkValue($item, $section)); ?>" placeholder="<?php echo $section === 'forms' ? 'https://example.com/form' : 'https://t.me/c/-1234567890/1234'; ?>">
                                                        <div class="form-text"><?php echo $section === 'forms' ? 'آدرس فرم یا لینک تلگرام را وارد کنید' : 'لینک پیام تلگرام'; ?></div>
                                                    </div>
                                                    <div class="mb-3 form-field-full" id="file_upload_edit<?php echo $item['id']; ?>" style="<?php echo (($item['item_type'] ?? 'file') === 'folder' || empty($item['file_path'])) ? 'display:none;' : ''; ?>">
                                                        <label class="form-label">فایل</label>
                                                        <?php if (!empty($item['file_path'])): ?>
                                                            <div class="mb-2 p-3 bg-light rounded shadow-sm">
                                                                <small class="text-muted d-block mb-2">
                                                                    <i class="fas fa-file me-1"></i>فایل فعلی: <?php echo htmlspecialchars(basename($item['file_path'])); ?>
                                                                </small>
                                                                <a href="../<?php echo htmlspecialchars($item['file_path']); ?>" target="_blank" class="btn btn-sm btn-primary">
                                                                    <i class="fas fa-download me-1"></i>دانلود فایل
                                                                </a>
                                                            </div>
                                                            <?php
                                                            $currentFileName = basename($item['file_path']);
                                                            $currentFileExtension = pathinfo($currentFileName, PATHINFO_EXTENSION);
                                                            $currentFileNameWithoutExt = pathinfo($currentFileName, PATHINFO_FILENAME);
                                                            ?>
                                                            <div class="mb-3">
                                                                <label class="form-label">نام فایل (بدون پسوند)</label>
                                                                <input type="text" name="file_name" class="form-control" value="<?php echo htmlspecialchars($currentFileNameWithoutExt); ?>" placeholder="نام فایل">
                                                                <div class="form-text">می‌توانید نام فایل فیزیکی در سرور را تغییر دهید. پسوند فایل (<?php echo htmlspecialchars($currentFileExtension); ?>) به صورت خودکار اضافه می‌شود.</div>
                                                            </div>
                                                        <?php endif; ?>
                                                        <input type="file" name="file" class="form-control">
                                                        <div class="form-text">فایل جدید (حداکثر ۱۲۸ مگابایت) می‌تواند جایگزین فایل قبلی شود (اختیاری)</div>
                                                        <div class="upload-progress d-none mt-3">
                                                            <div class="progress">
                                                                <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%;">0%</div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <?php else: ?>
                                                    <div class="mb-3 form-field-full">
                                                        <label class="form-label">لینک</label>
                                                        <input type="url" name="telegram_message_link" class="form-control" value="<?php echo htmlspecialchars($item['telegram_message_link'] ?? ''); ?>" placeholder="https://t.me/c/-1234567890/1234">
                                                        <div class="form-text">لینک پیام تلگرام برای فوروارد (اختیاری)</div>
                                                    </div>
                                                    <div class="mb-3 form-field-full">
                                                        <label class="form-label">فایل</label>
                                                        <?php if (!empty($item['file_path'])): ?>
                                                            <div class="mb-2 p-3 bg-light rounded shadow-sm">
                                                                <small class="text-muted">
                                                                    <i class="fas fa-file me-1"></i>فایل فعلی: <?php echo htmlspecialchars($item['file_path']); ?>
                                                                </small>
                                                            </div>
                                                        <?php endif; ?>
                                                        <input type="file" name="file" class="form-control">
                                                        <div class="form-text">فایل جدید (حداکثر ۱۲۸ مگابایت) می‌تواند جایگزین فایل قبلی شود (اختیاری)</div>
                                                        <div class="upload-progress d-none mt-3">
                                                            <div class="progress">
                                                                <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%;">0%</div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php endif; ?>
                                            <?php if ($section !== 'ethics'): ?>
                                            <div class="mb-3 form-field-full">
                                                <label class="form-label">برچسب‌ها</label>
                                                <input type="text" name="tags" class="form-control" value="<?php echo htmlspecialchars(implode(', ', $tagsByItem[$item['id']] ?? [])); ?>" placeholder="مثلاً: بتن، ایمنی، آیین‌نامه">
                                                <div class="form-text">حداکثر ۱۰ برچسب؛ برچسب‌ها را با ویرگول یا خط جدید از هم جدا کنید.</div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <?php if (in_array($section, $fileBasedSections, true)): ?>
                                                <div class="form-row-2">
                                                    <div class="mb-3 form-field-full">
                                                        <label class="form-label">
                                                            <i class="fas fa-folder-open me-2"></i>جابجایی به پوشه
                                                        </label>
                                                        <select name="parent_id" class="form-control">
                                                            <option value="">— بدون پوشه (سطح اصلی) —</option>
                                                            <?php 
                                                            // دریافت همه فولدرها از section
                                                            $allFolders = [];
                                                            if (isset($allItemsForSearch)) {
                                                                $itemId = (int)$item['id'];
                                                                $allFolders = array_filter($allItemsForSearch, function($it) use ($itemId) {
                                                                    $isFolder = ($it['item_type'] ?? 'file') === 'folder';
                                                                    $folderId = (int)$it['id'];
                                                                    // جلوگیری از انتخاب خود آیتم
                                                                    if ($folderId === $itemId) {
                                                                        return false;
                                                                    }
                                                                    return $isFolder;
                                                                });
                                                            }
                                                            // مرتب‌سازی بر اساس عنوان
                                                            usort($allFolders, function($a, $b) {
                                                                return strcmp($a['title'] ?? '', $b['title'] ?? '');
                                                            });
                                                            $currentParentId = !empty($item['parent_id']) ? (int)$item['parent_id'] : '';
                                                            foreach ($allFolders as $folder): 
                                                            ?>
                                                                <option value="<?php echo $folder['id']; ?>" <?php echo $currentParentId === (int)$folder['id'] ? 'selected' : ''; ?>>
                                                                    <?php echo htmlspecialchars($folder['title']); ?>
                                                                </option>
                                                            <?php endforeach; ?>
                                                        </select>
                                                        <div class="form-text">می‌توانید این مطلب را به یک پوشه دیگر منتقل کنید</div>
                                                    </div>
                                                </div>
                                            <?php elseif ($section !== 'ethics'): ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">لغو</button>
                                            <button type="submit" class="btn btn-primary">ذخیره تغییرات</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    
                <?php else: ?>
                    <!-- داشبورد اصلی -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2 class="mb-0">داشبورد</h2>
                    </div>
                    
                    <!-- جستجوی سراسری در داشبورد -->
                    <div class="card mb-4 border-0 shadow-sm" style="background: #ffffff; border-radius: 16px; border: 1px solid #e5e7eb;">
                        <div class="card-body p-4">
                            <div class="d-flex align-items-center mb-3">
                                <div class="flex-shrink-0 me-3">
                                    <div style="width: 50px; height: 50px; margin-left: 10px; background: #f3f4f6; border-radius: 12px; display: flex; align-items: center; justify-content: center;">
                                        <i class="fas fa-search text-primary" style="font-size: 1.5rem;"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1">
                                    <h4 class="mb-1" style="font-weight: 700; color: #1f2937; font-size: 1.25rem;">جستجوی سراسری</h4>
                                    <p class="mb-0 text-muted" style="font-size: 0.9rem;">جستجو در همه بخش‌ها و مطالب</p>
                                </div>
                            </div>
                            <form method="GET" class="row g-3 align-items-end">
                                <div class="col-md-10">
                                    <div class="input-group input-group-lg" style="border-radius: 12px; border: 2px solid #e5e7eb; overflow: hidden; transition: all 0.3s;" onfocus="this.style.borderColor='#4f46e5';" onblur="this.style.borderColor='#e5e7eb';">
                                        <span class="input-group-text bg-white border-0" style="padding: 15px 18px;">
                                            <i class="fas fa-search text-muted" style="font-size: 1.2rem; margin-left: 0.75rem;"></i>
                                        </span>
                                        <input type="text" name="global_search" class="form-control form-control-lg border-0" 
                                               value="<?php echo htmlspecialchars($_GET['global_search'] ?? ''); ?>" 
                                               placeholder="جستجو در عنوان، توضیحات، تگ‌ها و محتوا..." 
                                               style="font-size: 1rem; padding: 15px 0;">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <button type="submit" class="btn btn-primary btn-lg w-100" style="font-weight: 600; border-radius: 12px; padding: 15px; transition: all 0.3s; white-space: nowrap; box-shadow: 0 2px 8px rgba(79, 70, 229, 0.2);">
                                        <i class="fas fa-search me-3"></i>جستجو
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <?php if (isset($_GET['global_search']) && !empty(trim($_GET['global_search']))): ?>
                        <!-- نتایج جستجوی سراسری -->
                        <?php
                        $globalSearchTerm = trim($_GET['global_search']);
                        $globalSearchResults = [];
                        $globalSearchTags = [];
                        
                        foreach ($sectionNames as $sec => $name) {
                            if (isset($models[$sec])) {
                                $results = $models[$sec]->getAllWithCreator();
                                foreach ($results as $item) {
                                    $title = $sec === 'faq' ? ($item['question'] ?? '') : ($item['title'] ?? '');
                                    $description = $item['description'] ?? '';
                                    $body = $item['body'] ?? '';
                                    
                                    // جستجو در عنوان
                                    $found = stripos($title, $globalSearchTerm) !== false;
                                    
                                    // جستجو در توضیحات
                                    if (!$found && !empty($description)) {
                                        $found = stripos($description, $globalSearchTerm) !== false;
                                    }
                                    
                                    // جستجو در بدنه خبر
                                    if (!$found && !empty($body)) {
                                        $found = stripos($body, $globalSearchTerm) !== false;
                                    }
                                    
                                    // جستجو در تگ‌ها
                                    if (!$found) {
                                        $itemTags = $tagModel->getTagsForItems($sec, [$item['id']]);
                                        if (!empty($itemTags[$item['id']] ?? [])) {
                                            foreach ($itemTags[$item['id']] as $tag) {
                                                if (stripos($tag, $globalSearchTerm) !== false) {
                                                    $found = true;
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                    
                                    if ($found) {
                                        $item['section'] = $sec;
                                        $item['section_name'] = $name;
                                        $globalSearchResults[] = $item;
                                        
                                        // دریافت تگ‌ها برای این آیتم
                                        $itemTags = $tagModel->getTagsForItems($sec, [$item['id']]);
                                        $globalSearchTags[$item['id']] = $itemTags[$item['id']] ?? [];
                                    }
                                }
                            }
                        }
                        ?>
                        <div class="card mb-4 border-0 shadow-sm" style="border: 1px solid #e5e7eb; border-radius: 16px;">
                            <div class="card-header bg-white" style="border-bottom: 2px solid #e5e7eb; border-radius: 16px 16px 0 0; padding: 20px;">
                                <div class="d-flex justify-content-between align-items-center">
                                    <h5 class="mb-0" style="font-weight: 700; color: #1f2937;">
                                        <i class="fas fa-search me-2 text-primary"></i>
                                        نتایج جستجو برای: "<strong><?php echo htmlspecialchars($globalSearchTerm); ?></strong>"
                                    </h5>
                                    <span class="badge bg-primary text-white shadow-sm" style="font-size: 1rem; padding: 10px 15px; border-radius: 20px; font-weight: 600;">
                                        <i class="fas fa-list me-1"></i><?php echo count($globalSearchResults); ?> نتیجه
                                    </span>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <?php if (empty($globalSearchResults)): ?>
                                    <div class="alert alert-info m-4 mb-0 border-0 shadow-sm" style="border-radius: 12px;">
                                        <div class="text-center py-5">
                                            <i class="fas fa-search fa-4x text-muted mb-3" style="opacity: 0.5;"></i>
                                            <h5 class="mb-2">نتیجه‌ای یافت نشد</h5>
                                            <p class="text-muted mb-0">لطفاً کلمه کلیدی دیگری را امتحان کنید</p>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover mb-0">
                                            <thead class="table-light" style="background: #f8f9fa;">
                                                <tr>
                                                    <th style="width: 120px; font-weight: 600;">بخش</th>
                                                    <th style="font-weight: 600;">عنوان</th>
                                                    <th style="width: 150px; font-weight: 600;">ایجاد شده توسط</th>
                                                    <th style="width: 150px; font-weight: 600;">تاریخ ایجاد</th>
                                                    <th style="width: 80px; font-weight: 600;">عملیات</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($globalSearchResults as $result): ?>
                                                    <tr style="cursor: pointer; transition: all 0.2s;" 
                                                        onclick="window.location.href='?section=<?php echo $result['section']; ?>&edit_id=<?php echo $result['id']; ?>'"
                                                        oncontextmenu="showDashboardContextMenu(event, '<?php echo $result['section']; ?>', <?php echo $result['id']; ?>); return false;"
                                                        class="table-row-hover"
                                                        onmouseenter="this.style.backgroundColor='#f0f4ff'; this.style.transform='translateX(-3px)';"
                                                        onmouseleave="this.style.backgroundColor=''; this.style.transform='';">
                                                        <td>
                                                            <span class="badge bg-primary shadow-sm" style="font-size: 0.9rem; padding: 8px 12px; border-radius: 8px;">
                                                                <?php echo htmlspecialchars($result['section_name']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <div>
                                                                <strong style="color: #1f2937;"><?php 
                                                                    $displayTitle = $result['section'] === 'faq' ? ($result['question'] ?? '') : ($result['title'] ?? '');
                                                                    echo htmlspecialchars(mb_substr($displayTitle, 0, 60)) . (mb_strlen($displayTitle) > 60 ? '...' : '');
                                                                ?></strong>
                                                                <?php if (!empty($globalSearchTags[$result['id']] ?? [])): ?>
                                                                    <div class="mt-2">
                                                                        <?php foreach ($globalSearchTags[$result['id']] as $tag): ?>
                                                                            <span class="badge bg-secondary text-light me-1 shadow-sm" style="font-size: 0.75rem; padding: 4px 8px; border-radius: 6px;">
                                                                                #<?php echo htmlspecialchars($tag); ?>
                                                                            </span>
                                                                        <?php endforeach; ?>
                                                                    </div>
                                                                <?php endif; ?>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted" style="display: flex; align-items: center; gap: 5px;">
                                                                <i class="fas fa-user-circle"></i>
                                                                <?php echo htmlspecialchars($result['creator_name'] ?? 'نامشخص'); ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted" style="display: flex; align-items: center; gap: 5px;">
                                                                <i class="fas fa-calendar-alt"></i>
                                                                <?php echo formatDateTimeHtml($result['created_at']); ?>
                                                            </small>
                                                        </td>
                                                        <td onclick="event.stopPropagation();">
                                                            <a href="?section=<?php echo $result['section']; ?>&edit_id=<?php echo $result['id']; ?>" 
                                                               class="btn btn-sm btn-warning shadow-sm" 
                                                               onclick="event.stopPropagation();"
                                                               title="ویرایش"
                                                               style="border-radius: 8px;">
                                                                <i class="fas fa-edit"></i>
                                                            </a>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <div class="row">
                        <?php foreach ($sectionNames as $sec => $name): ?>
                            <div class="col-md-3">
                                <div class="stat-card text-center">
                                    <i class="fas fa-folder stat-icon"></i>
                                    <h3><?php echo $stats[$sec] ?? 0; ?></h3>
                                    <p><?php echo $name; ?></p>
                                    <a href="?section=<?php echo $sec; ?>" class="btn btn-outline-primary btn-sm">مشاهده</a>
                </div>
            </div>
                        <?php endforeach; ?>
        </div>
                    
                    <div class="mt-4">
                        <div class="row">
                            <div class="col-md-<?php echo $currentUser['role'] === 'super_admin' ? '4' : '6'; ?> mb-3">
                                <div class="stat-card text-center">
                                    <i class="fas fa-chart-bar stat-icon" style="color: #28a745;"></i>
                                    <h4>آمار و آنالیز</h4>
                                    <p>مشاهده آمار استفاده از بات</p>
                                    <a href="analytics.php" class="btn btn-success">مشاهده آمار</a>
    </div>
                            </div>
                            <?php if ($currentUser['role'] === 'super_admin'): ?>
                                <div class="col-md-4 mb-3">
                                    <div class="stat-card text-center">
                                        <i class="fas fa-bullhorn stat-icon" style="color: #ff9800;"></i>
                                        <h4>ارسال پیام همگانی</h4>
                                        <p>ارسال پیام به همه کاربران بات با یک کلیک</p>
                                        <a href="?section=broadcast" class="btn btn-warning text-white">ارسال پیام</a>
                                    </div>
                                </div>
                            <?php endif; ?>
                            <div class="col-md-<?php echo $currentUser['role'] === 'super_admin' ? '4' : '6'; ?> mb-3">
                                <div class="stat-card text-center">
                                    <i class="fas fa-key stat-icon" style="color: #dc3545;"></i>
                                    <h4>تغییر رمز عبور</h4>
                                    <p>تغییر رمز عبور حساب کاربری</p>
                                    <a href="?section=change_password" class="btn btn-danger">تغییر رمز عبور</a>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- Modal افزودن مطلب -->
                <?php if (!empty($section) && isset($models[$section])): ?>
                <div class="modal fade" id="addModal" tabindex="-1">
                    <div class="modal-dialog modal-lg">
            <div class="modal-content" style="position: relative;">
                <button type="button" class="modal-close-btn" data-bs-dismiss="modal" aria-label="بستن">
                    <i class="fas fa-times"></i>
                </button>
                        <form method="POST" enctype="multipart/form-data" accept-charset="UTF-8" data-upload-form="true">
                    <div class="modal-body">
                                    <input type="hidden" name="action" value="add">
                                    <input type="hidden" name="section" value="<?php echo $section; ?>">
                                    <?php if ($parentId !== null): ?>
                                    <input type="hidden" name="default_parent_id" value="<?php echo $parentId; ?>">
                                    <?php endif; ?>
                                    
                                    <?php if ($section === 'faq'): ?>
                            <div class="mb-3">
                                            <label class="form-label">سؤال</label>
                                            <input type="text" name="question" class="form-control" required>
                            </div>
                            <div class="mb-3">
                                            <label class="form-label">پاسخ</label>
                                            <textarea name="answer" class="form-control" rows="4" required></textarea>
                            </div>
                                    <?php elseif ($section === 'ai'): ?>
                            <div class="mb-3">
                                            <label class="form-label">عنوان</label>
                                            <input type="text" name="title" class="form-control" required>
                        </div>
                            <div class="mb-3">
                                            <label class="form-label">لینک وبسایت</label>
                                            <input type="url" name="website_url" class="form-control" placeholder="https://example.com">
                            </div>
                                    <?php elseif ($section === 'ethics'): ?>
                            <div class="mb-3">
                                            <label class="form-label">جمله</label>
                                            <textarea name="quote_text" class="form-control" rows="3" required></textarea>
                            </div>
                                    <?php else: ?>
                            <div class="form-row-2">
                                <div class="mb-3 form-field-full">
                                    <label class="form-label">عنوان</label>
                                    <input type="text" name="title" class="form-control" required>
                                </div>
                                <?php if ($section === 'news'): ?>
                                <div class="mb-3 form-field-full">
                                    <label class="form-label">بدنه خبر</label>
                                    <textarea name="body" class="form-control" rows="6" placeholder="متن کامل خبر را وارد کنید..."></textarea>
                                    <div class="form-text">متن کامل خبر که در بات نمایش داده می‌شود</div>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">تاریخ خبر (شمسی)</label>
                                    <div class="input-group">
                                        <input type="text" name="persian_date" id="persian_date_add" class="form-control" placeholder="1403/01/01">
                                        <button type="button" class="btn btn-outline-info" onclick="setTodayPersianDate('add')" title="تاریخ امروز">
                                            <i class="fas fa-calendar-day"></i> امروز
                                        </button>
                                    </div>
                                    <div class="form-text">تاریخ انتشار خبر به شمسی (مثال: 1403/01/01)</div>
                                </div>
                                <?php endif; ?>
                                <?php if (in_array($section, $fileBasedSections, true)): ?>
                                <input type="hidden" name="item_type" value="file">
                                <div class="mb-3 form-field-full">
                                    <label class="form-label">نوع محتوا</label>
                                    <select name="content_type" class="form-control form-select" id="content_type_add" onchange="window.toggleContentType('add')">
                                    <option value="file" selected>📄  آپلود فایل</option>
                                    <option value="link">🔗  لینک</option>
                                </select>
                                <div class="form-text">یکی از گزینه‌های بالا را انتخاب کنید</div>
                            </div>
                            <div class="mb-3 form-field-full" id="file_upload_add">
                                <label class="form-label">آپلود فایل</label>
                                <input type="file" name="file" class="form-control" id="file_input_add" onchange="updateFileNameField('add'); window.showUploadProgress('add');">
                                <div class="form-text">حداکثر حجم مجاز فایل ۱۲۸ مگابایت است.</div>
                                <div class="mb-3 mt-3">
                                    <label class="form-label">نام فایل (بدون پسوند) - اختیاری</label>
                                    <input type="text" name="file_name" class="form-control" id="file_name_add" placeholder="نام فایل">
                                    <div class="form-text">اگر خالی باشد، از عنوان مطلب استفاده می‌شود. پسوند فایل به صورت خودکار اضافه می‌شود.</div>
                                </div>
                                <div class="upload-progress d-none mt-3" id="upload_progress_add">
                                    <div class="progress" style="height: 25px; border-radius: 10px;">
                                        <div class="progress-bar progress-bar-striped progress-bar-animated bg-primary" role="progressbar" style="width: 0%; font-weight: bold; display: flex; align-items: center; justify-content: center;">0%</div>
                                    </div>
                                    <div class="mt-2 text-center text-muted" id="upload_status_add" style="font-size: 0.9rem;"></div>
                                </div>
                            </div>
                            <div class="mb-3 form-field-full" id="telegram_link_add" style="display: none;">
                                <label class="form-label">لینک <span class="text-danger">*</span></label>
                                <input type="url" name="telegram_message_link" id="telegram_message_link_add" class="form-control" placeholder="<?php echo $section === 'forms' ? 'https://example.com/form' : 'https://t.me/c/-1234567890/1234'; ?>" value="">
                                <div class="form-text"><?php echo $section === 'forms' ? 'آدرس فرم یا لینک تلگرام را وارد کنید' : 'لینک پیام تلگرام'; ?></div>
                            </div>
                            <!-- فیلد دسته والد مخفی شده - از طریق فولدرها قابل دسترسی است -->
                            <input type="hidden" name="parent_id" value="<?php echo $parentId !== null ? (int)$parentId : ''; ?>">
                            <?php endif; ?>
                                    <?php endif; ?>
                                    <?php if ($section !== 'ethics'): ?>
                            <div class="mb-3">
                                        <label class="form-label">برچسب‌ها</label>
                                        <input type="text" name="tags" class="form-control" placeholder="مثلاً: بتن، ایمنی، دستورالعمل">
                                        <div class="form-text">حداکثر ۱۰ برچسب؛ برچسب‌ها را با ویرگول یا خط جدید از هم جدا کنید.</div>
                                    </div>
                                    <?php endif; ?>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">لغو</button>
                                    <button type="submit" class="btn btn-primary">افزودن</button>
                    </div>
                </form>
                    </div>
        </div>
                </div>
                            </div>
                        </div>
                        
        <!-- Modal افزودن فولدر -->
        <?php if (!empty($section) && isset($models[$section]) && in_array($section, $fileBasedSections, true)): ?>
        <div class="modal fade" id="addFolderModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content" style="position: relative;">
                    <button type="button" class="modal-close-btn" data-bs-dismiss="modal" aria-label="بستن">
                        <i class="fas fa-times"></i>
                    </button>
                    <form method="POST" enctype="multipart/form-data">
                        <div class="modal-body">
                            <input type="hidden" name="action" value="add">
                            <input type="hidden" name="section" value="<?php echo $section; ?>">
                            <input type="hidden" name="item_type" value="folder">
                            <?php if ($parentId !== null): ?>
                            <input type="hidden" name="default_parent_id" value="<?php echo $parentId; ?>">
                            <?php endif; ?>
                            
                            <div class="mb-3 form-field-full">
                                <label class="form-label">عنوان فولدر</label>
                                <input type="text" name="title" class="form-control" required placeholder="نام فولدر را وارد کنید">
                                <div class="form-text">نام فولدر برای گروه‌بندی فایل‌ها</div>
                            </div>
                            
                            <!-- فیلد دسته والد مخفی شده - از طریق فولدرها قابل دسترسی است -->
                            <input type="hidden" name="parent_id" value="<?php echo $parentId !== null ? (int)$parentId : ''; ?>">
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">لغو</button>
                            <button type="submit" class="btn btn-success">افزودن فولدر</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
    <?php endif; ?>
    
    <!-- مودال‌های تغییر پسوورد -->
    <?php if (canManageAdmins() && !empty($allAdmins)): ?>
        <?php foreach ($allAdmins as $admin): ?>
            <?php if ($admin['id'] != $currentUser['id']): ?>
            <div class="modal fade" id="changePasswordModal<?php echo $admin['id']; ?>" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                            <h5 class="modal-title">تغییر رمز عبور - <?php echo htmlspecialchars($admin['username']); ?></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                        <form method="POST">
                <div class="modal-body">
                                <input type="hidden" name="action" value="change_password">
                                <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                                
                                <div class="mb-3">
                                    <label class="form-label">رمز عبور جدید</label>
                                    <input type="password" name="new_password" class="form-control" required minlength="6">
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">تأیید رمز عبور</label>
                                    <input type="password" name="confirm_password" class="form-control" required minlength="6">
                                </div>
                </div>
                <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">انصراف</button>
                                <button type="submit" class="btn btn-primary">تغییر رمز عبور</button>
                            </div>
                    </form>
                </div>
            </div>
        </div>
        <?php endif; ?>
    <?php endforeach; ?>
    <?php endif; ?>
    
    <!-- Footer -->
    <footer class="text-center py-3 bg-light border-top">
        <div class="container-fluid">
            <p class="mb-0 text-muted">
                <i class="fas fa-code"></i> توسعه‌دهندگان: <strong>سعید خلیلی</strong> + <a href="https://mhkstudio.ir/" target="_blank" rel="noopener noreferrer" style="text-decoration: underline; color: #007bff; font-weight: bold;"><strong>محمدحسین خانی</strong></a>
            </p>
    </div>
    </footer>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <link rel="stylesheet" href="https://unpkg.com/persian-datepicker@1.2.0/dist/css/persian-datepicker.min.css">
    <script src="https://unpkg.com/persian-date@1.1.0/dist/persian-date.min.js"></script>
    <script src="https://unpkg.com/persian-datepicker@1.2.0/dist/js/persian-datepicker.min.js"></script>
    <script>
        // اطمینان از تعریف تابع toggleContentType
        if (typeof window.toggleContentType === 'undefined') {
            window.toggleContentType = function(mode) {
                console.log('toggleContentType called with mode:', mode);
                
                var selectElement = document.getElementById('content_type_' + mode);
                if (!selectElement) {
                    console.error('Select element not found: content_type_' + mode);
                    setTimeout(function() {
                        window.toggleContentType(mode);
                    }, 100);
                    return;
                }
                
                var contentType = selectElement.value;
                var telegramLink = document.getElementById('telegram_link_' + mode);
                var fileUpload = document.getElementById('file_upload_' + mode);
                var telegramInput = document.getElementById('telegram_message_link_' + mode);
                var fileInput = document.getElementById('file_input_' + mode);
                
                if (contentType === 'link') {
                    if (telegramLink) {
                        telegramLink.style.setProperty('display', 'block', 'important');
                        telegramLink.style.setProperty('visibility', 'visible', 'important');
                    }
                    if (telegramInput) {
                        telegramInput.setAttribute('required', 'required');
                    }
                    if (fileUpload) {
                        fileUpload.style.setProperty('display', 'none', 'important');
                        fileUpload.style.setProperty('visibility', 'hidden', 'important');
                    }
                    if (fileInput) {
                        fileInput.removeAttribute('required');
                    }
                } else {
                    if (telegramLink) {
                        telegramLink.style.setProperty('display', 'none', 'important');
                        telegramLink.style.setProperty('visibility', 'hidden', 'important');
                    }
                    if (telegramInput) {
                        telegramInput.removeAttribute('required');
                    }
                    if (fileUpload) {
                        fileUpload.style.setProperty('display', 'block', 'important');
                        fileUpload.style.setProperty('visibility', 'visible', 'important');
                    }
                    if (fileInput) {
                        fileInput.setAttribute('required', 'required');
                    }
                }
            };
        }
        
        function toggleContentTypeEdit(itemId) {
            var contentType = document.getElementById('content_type_edit' + itemId).value;
            var telegramLink = document.getElementById('telegram_link_edit' + itemId);
            var fileUpload = document.getElementById('file_upload_edit' + itemId);
            
            if (contentType === 'link') {
                if (telegramLink) telegramLink.style.setProperty('display', 'block', 'important');
                if (fileUpload) fileUpload.style.setProperty('display', 'none', 'important');
            } else {
                if (telegramLink) telegramLink.style.setProperty('display', 'none', 'important');
                if (fileUpload) fileUpload.style.setProperty('display', 'block', 'important');
            }
        }
        
        // تنظیم state اولیه هنگام باز شدن modal افزودن
        document.addEventListener('DOMContentLoaded', function() {
            // استفاده از event delegation برای modal
            $(document).on('shown.bs.modal', '#addModal', function() {
                setTimeout(function() {
                    window.toggleContentType('add');
                }, 100);
            });
            
            // استفاده از event delegation برای تغییر select
            $(document).on('change', '#content_type_add', function() {
                window.toggleContentType('add');
            });
            
            // همچنین برای اطمینان، هنگام کلیک روی دکمه افزودن مطلب
            $(document).on('click', '[data-bs-target="#addModal"]', function() {
                setTimeout(function() {
                    window.toggleContentType('add');
                }, 200);
            });
        });
        
        function toggleFileFieldsEdit(itemId) {
            var itemType = document.getElementById('item_type_edit' + itemId).value;
            var fileFields = document.getElementById('file_fields_edit' + itemId);
            var telegramLink = document.getElementById('telegram_link_edit' + itemId);
            var fileUpload = document.getElementById('file_upload_edit' + itemId);
            
            if (itemType === 'folder') {
                if (fileFields) fileFields.style.display = 'none';
                if (telegramLink) telegramLink.style.display = 'none';
                if (fileUpload) fileUpload.style.display = 'none';
            } else {
                if (fileFields) fileFields.style.display = 'block';
                // نمایش فیلد مناسب بر اساس content_type
                var contentTypeSelect = document.getElementById('content_type_edit' + itemId);
                if (contentTypeSelect) {
                    var contentType = contentTypeSelect.value;
                    if (contentType === 'link') {
                        if (telegramLink) telegramLink.style.display = 'block';
                        if (fileUpload) fileUpload.style.display = 'none';
                } else {
                        if (telegramLink) telegramLink.style.display = 'none';
                        if (fileUpload) fileUpload.style.display = 'block';
                    }
                }
            }
        }
        
        function updateFileNameField(mode) {
            var fileInput = document.getElementById('file_input_' + mode);
            var fileNameField = document.getElementById('file_name_' + mode);
            
            if (fileInput && fileNameField && fileInput.files && fileInput.files.length > 0) {
                var fileName = fileInput.files[0].name;
                // استخراج نام فایل بدون پسوند
                var lastDotIndex = fileName.lastIndexOf('.');
                var fileNameWithoutExt = lastDotIndex > 0 ? fileName.substring(0, lastDotIndex) : fileName;
                
                // فقط اگر فیلد خالی باشد، نام فایل را قرار می‌دهیم
                if (!fileNameField.value || fileNameField.value.trim() === '') {
                    fileNameField.value = fileNameWithoutExt;
                }
            }
        }
        
        window.showUploadProgress = function(mode) {
            var fileInput = document.getElementById('file_input_' + mode);
            var progressWrapper = document.getElementById('upload_progress_' + mode);
            var progressBar = progressWrapper ? progressWrapper.querySelector('.progress-bar') : null;
            var statusText = progressWrapper ? document.getElementById('upload_status_' + mode) : null;
            
            if (fileInput && fileInput.files && fileInput.files.length > 0 && progressWrapper && progressBar) {
                var file = fileInput.files[0];
                var fileSize = (file.size / (1024 * 1024)).toFixed(2);
                
                progressWrapper.classList.remove('d-none');
                progressBar.style.width = '0%';
                progressBar.textContent = '0%';
                progressBar.classList.remove('bg-danger', 'bg-success');
                progressBar.classList.add('bg-primary', 'progress-bar-animated');
                
                if (statusText) {
                    statusText.textContent = 'فایل انتخاب شد: ' + file.name + ' (' + fileSize + ' MB) - آماده برای آپلود';
                    statusText.classList.remove('text-danger', 'text-success');
                    statusText.classList.add('text-muted');
                }
            }
        };
        
        function openChangePasswordModal(adminId) {
            console.log('Opening modal for admin ID:', adminId);
            var modalElement = document.getElementById('changePasswordModal' + adminId);
            console.log('Modal element:', modalElement);
            if (modalElement) {
                var modal = new bootstrap.Modal(modalElement);
                modal.show();
                } else {
                alert('مودال پیدا نشد!');
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            var uploadForms = document.querySelectorAll('form[data-upload-form="true"]');
            uploadForms.forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    var fileInputs = form.querySelectorAll('input[type="file"]');
                    var hasFile = false;
                    fileInputs.forEach(function(input) {
                        if (input.files && input.files.length > 0) {
                            hasFile = true;
                        }
                    });

                    if (!hasFile) {
                        return;
                    }

                    event.preventDefault();

                    var submitButtons = form.querySelectorAll('button[type="submit"]');
                    submitButtons.forEach(function(btn) {
                        btn.disabled = true;
                    });

                    // پیدا کردن progress wrapper - ابتدا با ID، سپس با class
                    var formId = form.id || '';
                    var isAddForm = formId.includes('add') || form.querySelector('input[name="action"][value="add"]');
                    var itemId = form.querySelector('input[name="id"]')?.value || '';
                    var progressId = isAddForm ? 'upload_progress_add' : ('upload_progress_edit' + itemId);
                    var progressWrapper = document.getElementById(progressId) || form.querySelector('.upload-progress');
                    var progressBar = progressWrapper ? progressWrapper.querySelector('.progress-bar') : null;
                    var statusId = isAddForm ? 'upload_status_add' : ('upload_status_edit' + itemId);
                    var statusText = document.getElementById(statusId) || (progressWrapper ? progressWrapper.querySelector('[id^="upload_status"]') : null);
                    
                    console.log('Form ID:', formId, 'Is Add:', isAddForm, 'Item ID:', itemId);
                    console.log('Progress ID:', progressId, 'Progress wrapper:', progressWrapper, 'Progress bar:', progressBar, 'Status text:', statusText);
                    
                    if (progressWrapper && progressBar) {
                        progressWrapper.classList.remove('d-none');
                        progressWrapper.style.display = 'block';
                        progressWrapper.style.visibility = 'visible';
                        progressBar.classList.remove('bg-danger', 'bg-success');
                        progressBar.classList.add('bg-primary');
                        if (!progressBar.classList.contains('progress-bar-animated')) {
                            progressBar.classList.add('progress-bar-animated');
                        }
                        progressBar.style.width = '0%';
                        progressBar.textContent = '0%';
                        if (statusText) {
                            statusText.textContent = 'در حال آماده‌سازی...';
                            statusText.style.display = 'block';
                            statusText.style.visibility = 'visible';
                        }
                    } else {
                        console.error('Progress bar elements not found! Progress ID:', progressId);
                    }

                    var xhr = new XMLHttpRequest();
                    xhr.open(form.method || 'POST', form.getAttribute('action') || window.location.href, true);
                    xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

                    xhr.upload.addEventListener('progress', function(e) {
                        if (!progressBar) {
                            return;
                        }
                        if (e.lengthComputable) {
                            var percent = Math.round((e.loaded / e.total) * 100);
                            var loadedMB = (e.loaded / (1024 * 1024)).toFixed(2);
                            var totalMB = (e.total / (1024 * 1024)).toFixed(2);
                            
                            progressBar.style.width = percent + '%';
                            progressBar.textContent = percent + '%';
                            
                            if (statusText) {
                                statusText.textContent = 'در حال آپلود: ' + loadedMB + ' MB از ' + totalMB + ' MB';
                            }
                        } else {
                            progressBar.style.width = '50%';
                            progressBar.textContent = 'در حال آپلود...';
                            if (statusText) {
                                statusText.textContent = 'در حال آپلود فایل...';
                            }
                        }
                    });

                    var handleError = function() {
                        if (progressBar) {
                            progressBar.classList.remove('progress-bar-animated');
                            progressBar.classList.remove('bg-primary');
                            progressBar.classList.add('bg-danger');
                            progressBar.textContent = 'خطا در آپلود';
                        }
                        if (statusText) {
                            statusText.textContent = 'خطا در آپلود فایل. لطفاً دوباره تلاش کنید.';
                            statusText.classList.add('text-danger');
                        }
                        submitButtons.forEach(function(btn) {
                            btn.disabled = false;
                        });
                        showNotification('خطایی در آپلود فایل رخ داد. لطفاً دوباره تلاش کنید.', 'error');
                    };

                    xhr.addEventListener('error', handleError);
                    xhr.addEventListener('abort', handleError);

                    xhr.addEventListener('load', function() {
                        if (xhr.status >= 200 && xhr.status < 300) {
                            if (progressBar) {
                                progressBar.style.width = '100%';
                                progressBar.textContent = '100%';
                                progressBar.classList.remove('progress-bar-animated', 'bg-danger', 'bg-primary');
                                progressBar.classList.add('bg-success');
                            }
                            if (statusText) {
                                statusText.textContent = 'آپلود با موفقیت انجام شد!';
                                statusText.classList.remove('text-danger');
                                statusText.classList.add('text-success');
                            }
                            
                            // نمایش پیام موفقیت
                            try {
                                var response = JSON.parse(xhr.responseText);
                                if (response.success && response.message) {
                                    showNotification(response.message, 'success');
                                } else {
                                    showNotification('مطلب با موفقیت اضافه شد', 'success');
                                }
                            } catch (e) {
                                showNotification('مطلب با موفقیت اضافه شد', 'success');
                            }
                            
                            setTimeout(function() {
                                window.location.reload();
                            }, 1500);
                        } else {
                            handleError();
                        }
                    });

                    var formData = new FormData(form);
                    xhr.send(formData);
                });
            });
        });
        
        // تقویم شمسی
        window.openPersianCalendar = function(mode) {
            var inputId = 'persian_date_' + mode;
            var input = document.getElementById(inputId);
            if (!input) {
                console.error('Input not found: ' + inputId);
                alert('فیلد تاریخ یافت نشد. لطفاً صفحه را رفرش کنید.');
                return;
            }
            
            console.log('Opening calendar for:', inputId, 'Input element:', input);
            
            // بررسی لود شدن jQuery و datepicker
            if (typeof $ === 'undefined') {
                console.error('jQuery is not loaded');
                alert('خطا: کتابخانه jQuery لود نشده است. لطفاً صفحه را رفرش کنید.');
                return;
            }
            
            // بررسی لود شدن persian-datepicker
            if (typeof $.fn === 'undefined' || typeof $.fn.pDatepicker === 'undefined') {
                console.error('pDatepicker is not loaded, waiting...');
                // تلاش برای لود مجدد
                setTimeout(function() {
                    if (typeof $.fn !== 'undefined' && typeof $.fn.pDatepicker !== 'undefined') {
                        window.openPersianCalendar(mode);
                    } else {
                        console.error('pDatepicker still not loaded after timeout');
                        alert('خطا: تقویم شمسی لود نشده است. لطفاً صفحه را رفرش کنید.');
                    }
                }, 1000);
                return;
            }
            
            // تلاش برای استفاده از persian-datepicker
            try {
                // اگر قبلاً datepicker روی این input تنظیم شده، آن را حذف می‌کنیم
                if ($(input).data('pDatepicker')) {
                    $(input).pDatepicker('destroy');
                }
                
                console.log('Initializing pDatepicker for:', inputId);
                
                // مقدار اولیه را تنظیم می‌کنیم
                var initialValue = input.value || false;
                
                $(input).pDatepicker({
                    format: 'YYYY/MM/DD',
                    calendarType: 'persian',
                    initialValue: initialValue,
                    autoClose: true,
                    observer: true,
                    altField: '',
                    altFormat: 'YYYY/MM/DD',
                    timePicker: {
                        enabled: false
                    },
                    zIndex: 10000,
                    onSelect: function(date) {
                        if (date) {
                            var year = date.year;
                            var month = String(date.month).padStart(2, '0');
                            var day = String(date.day).padStart(2, '0');
                            input.value = year + '/' + month + '/' + day;
                            $(input).trigger('change');
                            console.log('Date selected:', input.value);
                        }
                    }
                });
                
                console.log('pDatepicker initialized, showing calendar...');
                
                // نمایش تقویم با تاخیر
                setTimeout(function() {
                    try {
                        console.log('Attempting to show datepicker for:', inputId);
                        // اطمینان از اینکه input focus شده باشد
                        $(input).focus();
                        // نمایش تقویم
                        $(input).pDatepicker('show');
                        console.log('Datepicker shown successfully');
                    } catch (e) {
                        console.error('Error showing datepicker:', e);
                        // تلاش مجدد
                        setTimeout(function() {
                            try {
                                $(input).focus();
                                $(input).pDatepicker('show');
                            } catch (e2) {
                                console.error('Second attempt failed:', e2);
                                window.fallbackDateInput(input);
                            }
                        }, 500);
                    }
                }, 300);
            } catch (e) {
                console.error('Error initializing datepicker:', e);
                window.fallbackDateInput(input);
            }
        };
        
        // تابع fallback برای ورود تاریخ
        window.fallbackDateInput = function(input) {
            var currentDate = input.value || '';
            var newDate = prompt('تاریخ شمسی را وارد کنید (مثال: 1403/01/01):', currentDate);
            if (newDate && /^\d{4}\/\d{2}\/\d{2}$/.test(newDate)) {
                input.value = newDate;
                $(input).trigger('change');
            }
        }
        
        window.setTodayDate = function(mode) {
            var inputId = 'persian_date_' + mode;
            var input = document.getElementById(inputId);
            if (!input) return;
            
            // محاسبه تاریخ امروز شمسی
            try {
                if (typeof persianDate !== 'undefined') {
                    var today = persianDate();
                    var year = today.year();
                    var month = String(today.month()).padStart(2, '0');
                    var day = String(today.date()).padStart(2, '0');
                    input.value = year + '/' + month + '/' + day;
                } else {
                    // درخواست از سرور برای تاریخ امروز شمسی
                    fetch('helpers/get_today_persian.php')
                        .then(response => response.json())
                        .then(data => {
                            if (data.success && data.date) {
                                input.value = data.date;
                            } else {
                                // Fallback
                                var today = new Date();
                                input.value = '1403/01/01';
                            }
                        })
                        .catch(error => {
                            console.error('Error fetching today date:', error);
                            input.value = '1403/01/01';
                        });
                }
            } catch (e) {
                console.error('Error setting today date:', e);
                input.value = '1403/01/01';
            }
        };
        
        // تابع برای وارد کردن تاریخ امروز شمسی در فیلد تاریخ خبر
        window.setTodayPersianDate = function(mode) {
            var inputId = 'persian_date_' + mode;
            var input = document.getElementById(inputId);
            if (!input) return;
            
            // درخواست از سرور برای تاریخ امروز شمسی
            fetch('helpers/get_today_persian.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.date) {
                        input.value = data.date;
                        // نمایش پیام موفقیت موقت
                        var originalPlaceholder = input.placeholder;
                        input.style.borderColor = '#28a745';
                        setTimeout(function() {
                            input.style.borderColor = '';
                        }, 1000);
                    } else {
                        console.error('Failed to get today date');
                    }
                })
                .catch(error => {
                    console.error('Error fetching today date:', error);
                });
        };
        
        // سیستم نوتیفیکیشن داینامیک
        function showNotification(message, type = 'success') {
            const container = document.getElementById('notificationContainer');
            if (!container) {
                console.error('Notification container not found!');
                return;
            }
            
            const notification = document.createElement('div');
            notification.className = 'dynamic-notification ' + type;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            const iconClass = icons[type] || icons.success;
            
            notification.innerHTML = 
                '<i class="fas ' + iconClass + ' notification-icon"></i>' +
                '<span class="notification-content">' + message + '</span>' +
                '<button class="notification-close" onclick="this.parentElement.remove()">' +
                '<i class="fas fa-times"></i>' +
                '</button>';
            
            container.appendChild(notification);
            
            // حذف خودکار بعد از 5 ثانیه
            setTimeout(function() {
                notification.style.animation = 'slideUp 0.3s ease-in forwards';
                setTimeout(function() {
                    if (notification.parentElement) {
                        notification.remove();
                    }
                }, 300);
            }, 5000);
        }
        
        // نمایش نوتیفیکیشن‌های PHP
        <?php if (!empty($success_message)): ?>
        (function() {
            function showSuccessNotification() {
                var container = document.getElementById('notificationContainer');
                if (container) {
                    showNotification(<?php echo json_encode($success_message, JSON_UNESCAPED_UNICODE | JSON_HEX_APOS); ?>, 'success');
                } else {
                    setTimeout(showSuccessNotification, 100);
                }
            }
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', showSuccessNotification);
            } else {
                showSuccessNotification();
            }
        })();
        <?php endif; ?>
        
        <?php if (!empty($error_message)): ?>
        (function() {
            function showErrorNotification() {
                var container = document.getElementById('notificationContainer');
                if (container) {
                    showNotification(<?php echo json_encode($error_message, JSON_UNESCAPED_UNICODE | JSON_HEX_APOS); ?>, 'error');
                } else {
                    setTimeout(showErrorNotification, 100);
                }
            }
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', showErrorNotification);
            } else {
                showErrorNotification();
            }
        })();
        <?php endif; ?>
        
        // جستجو در جدول (client-side برای سرعت بیشتر)
        document.addEventListener('DOMContentLoaded', function() {
            var searchInput = document.querySelector('input[name="search"]');
            if (searchInput) {
                var table = document.querySelector('.table');
                if (table) {
                    var originalRows = Array.from(table.querySelectorAll('tbody tr'));
                    
                    searchInput.addEventListener('input', function() {
                        var searchTerm = this.value.toLowerCase();
                        originalRows.forEach(function(row) {
                            var text = row.textContent.toLowerCase();
                            row.style.display = text.includes(searchTerm) ? '' : 'none';
                        });
                    });
                }
            }
            
            // بررسی لود شدن کتابخانه‌های تقویم
            if (typeof $ === 'undefined') {
                console.error('jQuery not loaded!');
            } else {
                console.log('jQuery loaded successfully');
            }
            
            if (typeof $.fn === 'undefined' || typeof $.fn.pDatepicker === 'undefined') {
                console.warn('persian-datepicker not loaded! Using fallback method.');
            } else {
                console.log('persian-datepicker loaded successfully');
            }
            
            // بستن منوی راست کلیک با کلیک در جای دیگر
            document.addEventListener('click', function(e) {
                var contextMenu = document.getElementById('contextMenu');
                if (contextMenu && !contextMenu.contains(e.target)) {
                    contextMenu.style.display = 'none';
                }
            });
            
            // راست کلیک برای سطرهای جدول
            document.addEventListener('contextmenu', function(e) {
                var row = e.target.closest('tbody tr');
                if (row) {
                    var itemId = row.getAttribute('data-item-id');
                    var section = row.getAttribute('data-section');
                    var isFolder = row.getAttribute('data-is-folder') === '1';
                    var isPinned = row.getAttribute('data-is-pinned') === '1';
                    
                    if (itemId && section) {
                        e.preventDefault();
                        e.stopPropagation();
                        showContextMenu(e, parseInt(itemId), section, isFolder, isPinned);
                        return false;
                    }
                }
            });
            
            // جلوگیری از نمایش منوی پیش‌فرض مرورگر
            document.addEventListener('contextmenu', function(e) {
                if (e.target.closest('tbody tr[data-item-id]')) {
                    e.preventDefault();
                    return false;
                }
            }, true);
        });
        
        // متغیرهای سراسری برای منوی راست کلیک
        var currentContextItemId = null;
        var currentContextSection = null;
        var currentContextIsFolder = false;
        var currentContextIsPinned = false;
        var currentContextParentId = null;
        
        // نمایش منوی راست کلیک در جدول
        function showContextMenu(event, itemId, section, isFolder, isPinned) {
            if (event) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            currentContextItemId = itemId;
            currentContextSection = section;
            currentContextIsFolder = isFolder;
            currentContextIsPinned = isPinned;
            currentContextParentId = <?php echo isset($parentId) && $parentId !== null ? $parentId : 'null'; ?>;
            
            var menu = document.getElementById('contextMenu');
            if (!menu) {
                console.error('Context menu not found!');
                return;
            }
            
            // موقعیت منو
            var x = event ? event.pageX : 0;
            var y = event ? event.pageY : 0;
            
            // اطمینان از اینکه منو خارج از صفحه نرود
            var menuWidth = 200;
            var menuHeight = 300;
            if (x + menuWidth > window.innerWidth) {
                x = window.innerWidth - menuWidth - 10;
            }
            if (y + menuHeight > window.innerHeight) {
                y = window.innerHeight - menuHeight - 10;
            }
            
            menu.style.display = 'block';
            menu.style.left = x + 'px';
            menu.style.top = y + 'px';
            menu.style.zIndex = '10000';
            
            // به روزرسانی متن دکمه پین
            var pinText = document.getElementById('contextMenuPinText');
            if (pinText) {
                pinText.textContent = isPinned ? 'برداشتن پین' : 'پین کردن';
            }
            
            // نمایش/مخفی کردن گزینه‌های مرتبط با فولدر
            var moveUpItem = menu.querySelector('[onclick*="move_up"]');
            var moveDownItem = menu.querySelector('[onclick*="move_down"]');
            if (moveUpItem && moveDownItem) {
                // فقط برای بخش‌های file-based نمایش بده
                var fileBasedSections = ['instructions', 'regulations', 'trainings', 'reports', 'news', 'forms'];
                if (fileBasedSections.includes(section)) {
                    moveUpItem.style.display = 'flex';
                    moveDownItem.style.display = 'flex';
                } else {
                    moveUpItem.style.display = 'none';
                    moveDownItem.style.display = 'none';
                }
            }
            
            // مخفی کردن پین برای ethics
            var pinItem = menu.querySelector('[onclick*="pin"]');
            if (pinItem && section === 'ethics') {
                pinItem.style.display = 'none';
            } else if (pinItem) {
                pinItem.style.display = 'flex';
            }
        }
        
        // نمایش منوی راست کلیک در داشبورد
        function showDashboardContextMenu(event, section, itemId) {
            if (event) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            currentContextItemId = itemId;
            currentContextSection = section;
            currentContextIsFolder = false;
            currentContextIsPinned = false;
            currentContextParentId = null;
            
            var menu = document.getElementById('contextMenu');
            if (!menu) {
                console.error('Context menu not found!');
                return;
            }
            
            menu.innerHTML = `
                <div class="context-menu-item" onclick="contextMenuAction('edit')">
                    <i class="fas fa-edit"></i>
                    <span>ویرایش</span>
                </div>
                <div class="context-menu-item" onclick="contextMenuAction('view_in_section')">
                    <i class="fas fa-folder-open"></i>
                    <span>مشاهده در بخش مربوطه</span>
                </div>
            `;
            
            // موقعیت منو
            var x = event ? event.pageX : 0;
            var y = event ? event.pageY : 0;
            
            // اطمینان از اینکه منو خارج از صفحه نرود
            var menuWidth = 200;
            var menuHeight = 150;
            if (x + menuWidth > window.innerWidth) {
                x = window.innerWidth - menuWidth - 10;
            }
            if (y + menuHeight > window.innerHeight) {
                y = window.innerHeight - menuHeight - 10;
            }
            
            menu.style.display = 'block';
            menu.style.left = x + 'px';
            menu.style.top = y + 'px';
            menu.style.zIndex = '10000';
        }
        
        // اجرای عملیات منوی راست کلیک
        function contextMenuAction(action) {
            var menu = document.getElementById('contextMenu');
            menu.style.display = 'none';
            
            if (!currentContextItemId || !currentContextSection) {
                return;
            }
            
            switch(action) {
                case 'pin':
                    // تغییر وضعیت پین
                    var form = document.createElement('form');
                    form.method = 'POST';
                    form.action = window.location.pathname + window.location.search.split('&action=')[0].split('?action=')[0];
                    form.innerHTML = `
                        <input type="hidden" name="action" value="toggle_pin">
                        <input type="hidden" name="section" value="${currentContextSection}">
                        <input type="hidden" name="id" value="${currentContextItemId}">
                        ${currentContextParentId ? `<input type="hidden" name="parent_id" value="${currentContextParentId}">` : ''}
                    `;
                    document.body.appendChild(form);
                    form.submit();
                    break;
                    
                case 'move_up':
                    var form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="move_order">
                        <input type="hidden" name="section" value="${currentContextSection}">
                        <input type="hidden" name="id" value="${currentContextItemId}">
                        <input type="hidden" name="direction" value="up">
                        ${currentContextParentId ? `<input type="hidden" name="parent_id" value="${currentContextParentId}">` : ''}
                    `;
                    document.body.appendChild(form);
                    form.submit();
                    break;
                    
                case 'move_down':
                    var form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="move_order">
                        <input type="hidden" name="section" value="${currentContextSection}">
                        <input type="hidden" name="id" value="${currentContextItemId}">
                        <input type="hidden" name="direction" value="down">
                        ${currentContextParentId ? `<input type="hidden" name="parent_id" value="${currentContextParentId}">` : ''}
                    `;
                    document.body.appendChild(form);
                    form.submit();
                    break;
                    
                case 'edit':
                    if (currentContextIsFolder) {
                        window.location.href = `?section=${currentContextSection}&parent_id=${currentContextItemId}`;
                    } else {
                        $('#editModal' + currentContextItemId).modal('show');
                    }
                    break;
                    
                case 'delete':
                    if (confirm('آیا مطمئن هستید؟')) {
                        var form = document.createElement('form');
                        form.method = 'POST';
                        form.innerHTML = `
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="section" value="${currentContextSection}">
                            <input type="hidden" name="id" value="${currentContextItemId}">
                        `;
                        document.body.appendChild(form);
                        form.submit();
                    }
                    break;
                    
                case 'view_in_section':
                    window.location.href = `?section=${currentContextSection}&edit_id=${currentContextItemId}`;
                    break;
            }
        }
    </script>
</body>
</html>