<?php
/**
 * Helper functions for date conversion (Gregorian to Jalali)
 */

// بررسی اینکه آیا composer autoload وجود دارد
$vendorPath = __DIR__ . '/../../vendor/autoload.php';
if (file_exists($vendorPath)) {
    require_once $vendorPath;
}

/**
 * تبدیل تاریخ میلادی به شمسی
 */
function toJalali($gregorianDate, $format = 'Y/m/d H:i') {
    if (empty($gregorianDate) || $gregorianDate === '0000-00-00 00:00:00' || $gregorianDate === null) {
        return '-';
    }
    
    try {
        // اگر کتابخانه morilog/jalali موجود است، از آن استفاده کن
        if (class_exists('Morilog\Jalali\Jalalian')) {
            $jalali = \Morilog\Jalali\Jalalian::fromDateTime(new DateTime($gregorianDate));
            return $jalali->format($format);
        } else {
            // اگر کتابخانه موجود نیست، از تابع ساده استفاده کن
            return gregorianToJalaliSimple($gregorianDate, $format);
        }
    } catch (Exception $e) {
        error_log("Date conversion error: " . $e->getMessage());
        return $gregorianDate; // در صورت خطا، تاریخ میلادی را برگردان
    }
}

/**
 * تبدیل ساده تاریخ میلادی به شمسی (الگوریتم صحیح و معتبر)
 */
function gregorianToJalaliSimple($gregorianDate, $format = 'Y/m/d H:i') {
    try {
        // ایجاد DateTime object
        $date = new DateTime($gregorianDate);
        // تبدیل به timezone ایران
        $date->setTimezone(new DateTimeZone('Asia/Tehran'));
        
        $gYear = (int)$date->format('Y');
        $gMonth = (int)$date->format('m');
        $gDay = (int)$date->format('d');
        $hour = $date->format('H');
        $minute = $date->format('i');
        
        // الگوریتم تبدیل میلادی به شمسی - استفاده از الگوریتم ساده و معتبر
        // این الگوریتم بر اساس الگوریتم استاندارد تبدیل تاریخ شمسی است
        
        if (function_exists('gregoriantojd')) {
            $julianDay = gregoriantojd($gMonth, $gDay, $gYear);
        } else {
            // محاسبه دستی روز جولین
            $a = (int)((14 - $gMonth) / 12);
            $y = $gYear + 4800 - $a;
            $m = $gMonth + 12 * $a - 3;
            $julianDay = $gDay + (int)((153 * $m + 2) / 5) + 365 * $y + (int)($y / 4) - (int)($y / 100) + (int)($y / 400) - 32045;
        }
        
        // روز جولین برای 22 مارس 622 میلادی (1 فروردین 1 شمسی) = 1948321
        $baseJulianDay = 1948321;
        $daysSinceBase = $julianDay - $baseJulianDay;
        
        // محاسبه سال شمسی با استفاده از چرخه 33 ساله
        $cycles33 = (int)($daysSinceBase / 12053);
        $remainingDays = $daysSinceBase % 12053;
        
        // شروع از سال 1 شمسی
        $jy = $cycles33 * 33 + 1;
        
        // محاسبه سال دقیق
        while ($remainingDays >= 365) {
            // بررسی سال کبیسه
            $leapYear = ($jy % 33 == 1 || $jy % 33 == 5 || $jy % 33 == 9 || $jy % 33 == 13 || 
                        $jy % 33 == 17 || $jy % 33 == 22 || $jy % 33 == 26 || $jy % 33 == 30);
            $yearDays = $leapYear ? 366 : 365;
            
            if ($remainingDays >= $yearDays) {
                $remainingDays -= $yearDays;
                $jy++;
            } else {
                break;
            }
        }
        
        // تعیین سال کبیسه
        $leap = ($jy % 33 == 1 || $jy % 33 == 5 || $jy % 33 == 9 || $jy % 33 == 13 || 
                $jy % 33 == 17 || $jy % 33 == 22 || $jy % 33 == 26 || $jy % 33 == 30);
        
        // روزهای هر ماه شمسی
        $j_days_in_month = [31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, ($leap ? 30 : 29)];
        
        // محاسبه ماه و روز
        $jm = 1;
        $daysDiff = $remainingDays + 1; // +1 چون از روز 1 شروع می‌شود
        for ($i = 0; $i < 12 && $daysDiff > $j_days_in_month[$i]; $i++) {
            $daysDiff -= $j_days_in_month[$i];
            $jm++;
        }
        $jd = $daysDiff;
        
        // فرمت‌بندی خروجی
        $replacements = [
            'Y' => (string)$jy,
            'y' => substr((string)$jy, -2),
            'm' => str_pad((string)$jm, 2, '0', STR_PAD_LEFT),
            'n' => (string)$jm,
            'd' => str_pad((string)$jd, 2, '0', STR_PAD_LEFT),
            'j' => (string)$jd,
            'H' => $hour,
            'i' => $minute,
            's' => $date->format('s')
        ];
        
        $result = $format;
        foreach ($replacements as $key => $value) {
            $result = str_replace($key, $value, $result);
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Date conversion error: " . $e->getMessage() . " | Date: " . $gregorianDate);
        return $gregorianDate;
    }
}

/**
 * فرمت تاریخ برای نمایش در جدول
 */
function formatDateForDisplay($date, $showTime = true) {
    if ($showTime) {
        return toJalali($date, 'Y/m/d H:i');
    } else {
        return toJalali($date, 'Y/m/d');
    }
}

/**
 * ایجاد خروجی HTML با چینش تاریخ سمت راست و ساعت سمت چپ
 */
function formatDateTimeHtml($gregorianDate, $includeSeconds = false) {
    if (empty($gregorianDate) || $gregorianDate === '0000-00-00 00:00:00' || $gregorianDate === null) {
        return '-';
    }

    $format = $includeSeconds ? 'Y/m/d H:i:s' : 'Y/m/d H:i';
    $jalali = toJalali($gregorianDate, $format);

    if ($jalali === '-' || empty($jalali)) {
        return '-';
    }

    $parts = preg_split('/\s+/', trim($jalali));
    $datePart = $parts[0] ?? '';
    $timePart = $parts[1] ?? '';

    if ($datePart === '' && $timePart === '') {
        return '-';
    }

    $dateHtml = htmlspecialchars($datePart, ENT_QUOTES, 'UTF-8');
    $timeHtml = $timePart !== '' ? htmlspecialchars($timePart, ENT_QUOTES, 'UTF-8') : '';

    $output = '<span class="jalali-datetime">';
    if ($dateHtml !== '') {
        $output .= '<span class="jalali-date">' . $dateHtml . '</span>';
    }
    if ($timeHtml !== '') {
        $output .= '<span class="jalali-time">' . $timeHtml . '</span>';
    }
    $output .= '</span>';

    return $output;
}
